<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * List Extensions Action - Discover what's installed
 *
 * @since  1.0.0
 */
class ListExtensionsAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  List of extensions
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        $type = $parameters['type'] ?? null;
        $enabled = $parameters['enabled'] ?? null;

        $query = $db->getQuery(true)
            ->select('extension_id, name, type, element, folder, enabled')
            ->from($db->quoteName('#__extensions'))
            ->order($db->quoteName('type') . ', ' . $db->quoteName('name'));

        if ($type) {
            $query->where($db->quoteName('type') . ' = ' . $db->quote($type));
        }

        if ($enabled !== null) {
            $query->where($db->quoteName('enabled') . ' = ' . (int) $enabled);
        }

        $db->setQuery($query);
        $extensions = $db->loadObjectList();

        // Group by type
        $grouped = [];
        foreach ($extensions as $ext) {
            if (!isset($grouped[$ext->type])) {
                $grouped[$ext->type] = [];
            }
            $grouped[$ext->type][] = $ext;
        }

        return [
            'success' => true,
            'total_count' => count($extensions),
            'extensions' => $extensions,
            'grouped_by_type' => $grouped,
            'types' => array_keys($grouped)
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'List all installed extensions (components, modules, plugins). ' .
               'Optional parameters: type (component/module/plugin/template), enabled (1/0). ' .
               'Use this to discover what features/extensions are available on the site.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}


