# AI Assistant - System Architecture

This document provides a technical overview of the AI Assistant plugin architecture, design decisions, and extension points.

## Table of Contents

1. [System Overview](#system-overview)
2. [Core Components](#core-components)
3. [Data Flow](#data-flow)
4. [Agent Reasoning Loop](#agent-reasoning-loop)
5. [Action System](#action-system)
6. [Security Model](#security-model)
7. [Extension Points](#extension-points)
8. [Performance Considerations](#performance-considerations)

---

## System Overview

The AI Assistant is built as a Joomla 5 system plugin that provides an agentic AI interface for managing site content. It follows modern PHP 8.2+ practices and Joomla 5's namespace-based architecture.

### Design Philosophy

1. **Agentic Behavior**: The AI acts autonomously, planning and executing multi-step tasks
2. **Safety First**: Review mode and permissions prevent unintended changes
3. **Extensibility**: Easy to add new actions and AI providers
4. **Transparency**: Comprehensive logging of all operations
5. **Modern Stack**: Leverages latest PHP and Joomla features

### Technology Stack

- **Backend**: PHP 8.2+ with Joomla 5 APIs
- **Frontend**: Vanilla JavaScript (ES6+), modern CSS
- **AI Providers**: OpenAI API, Anthropic API
- **Database**: MySQL/MariaDB for session and action logging
- **Communication**: AJAX with JSON payloads

---

## Core Components

### 1. Plugin Entry Point

**File**: `src/Extension/AiAssistant.php`

The main plugin class that:
- Subscribes to Joomla events
- Routes AJAX requests
- Provides console interface
- Manages authentication

```php
class AiAssistant extends CMSPlugin implements SubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise'   => 'onAfterInitialise',
            'onBeforeCompileHead' => 'onBeforeCompileHead',
        ];
    }
}
```

### 2. Agent Orchestrator

**File**: `src/Agent/AgentOrchestrator.php`

The "brain" of the system that implements the reasoning loop:

```
User Prompt → Analyze → Plan Actions → Execute → Verify → Report
                ↑                                              |
                └──────────── Iterate if needed ──────────────┘
```

**Key Methods**:
- `processPrompt()`: Main entry point for user requests
- `parseAgentResponse()`: Extracts structured actions from AI response
- `executeAction()`: Runs a single action
- `executeReviewedActions()`: Runs previously planned actions

**Conversation Management**:
- Maintains conversation history for context
- Builds system prompt with available actions
- Feeds execution results back to AI for next iteration

### 3. AI Providers

**Files**: 
- `src/AI/AIProviderInterface.php`
- `src/AI/OpenAIProvider.php`
- `src/AI/AnthropicProvider.php`

Abstraction layer for different AI services:

```php
interface AIProviderInterface
{
    public function chat(array $messages): string;
    public function getName(): string;
    public function validateCredentials(): bool;
}
```

**OpenAI Implementation**:
- Uses Chat Completions API
- Supports GPT-5, GPT-5 Mini, GPT-5 Nano, GPT-4o, GPT-4o Mini, GPT-4 Turbo, GPT-4
- Standard message format

**Anthropic Implementation**:
- Uses Messages API
- Supports Claude Sonnet 4.5, Claude Haiku 4.5, Claude 3.5 Sonnet, Haiku, Claude 3 Opus, Sonnet, Haiku
- Converts system message to Anthropic format

### 4. Action System

**Files**:
- `src/Actions/ActionInterface.php`
- `src/Actions/ActionRegistry.php`
- `src/Actions/Actions/*.php`

Modular action system where each action is a self-contained class:

```php
interface ActionInterface
{
    public function execute(array $parameters): array;
    public static function getDescription(): string;
    public static function getRequiredParameters(): array;
    public function validate(array $parameters): bool;
}
```

**Action Registry**:
- Central registration point for all actions
- Enforces permission checks
- Maps action names to classes

**Built-in Actions**:
- Read: article, module, menu, pagebuilder, site info
- Write: create/update article, update module, update pagebuilder
- Search: articles, modules

### 5. Logging System

**File**: `src/Logger/ActionLogger.php`

Comprehensive audit trail that records:
- User sessions (prompt, status, results)
- Individual actions (type, parameters, results, errors)
- Timestamps and user attribution

**Database Schema**:

```sql
aiassistant_sessions:
  - id (session identifier)
  - user_id
  - prompt (original request)
  - status (running/completed/error)
  - result (JSON)
  - started_at, ended_at

aiassistant_actions:
  - id
  - session_id (FK)
  - action_type
  - parameters (JSON)
  - result (JSON)
  - status (planned/executed/failed)
  - error
  - executed_at
```

### 6. Console UI

**Files**:
- `tmpl/console.php` (HTML structure)
- `media/js/console.js` (UI logic)
- `media/css/console.css` (styling)

Modern, responsive interface with:
- Real-time conversation view
- Action review modal
- History sidebar
- Loading states
- Error handling

**JavaScript Architecture**:
```javascript
const AIConsole = {
    currentSession: null,
    
    submitPrompt() { /* ... */ },
    displayResult() { /* ... */ },
    showReviewModal() { /* ... */ },
    executeActions() { /* ... */ },
    apiRequest() { /* ... */ }
};
```

---

## Data Flow

### User Request Flow

```
1. User enters prompt in console
   ↓
2. JavaScript sends AJAX POST to plugin
   ↓
3. Plugin routes to processPrompt()
   ↓
4. AgentOrchestrator.processPrompt()
   ↓
5. Build conversation with system prompt
   ↓
6. Send to AI Provider (OpenAI/Anthropic)
   ↓
7. AI responds with planned actions
   ↓
8. Parse response → extract actions
   ↓
9. If review mode:
   - Store actions in database
   - Return to UI for approval
   Else:
   - Execute actions immediately
   ↓
10. Return results to UI
    ↓
11. Display to user
```

### Action Execution Flow

```
1. Action definition received
   ↓
2. ActionRegistry.getAction(type)
   ↓
3. Check permissions
   ↓
4. Instantiate action class
   ↓
5. action.validate(parameters)
   ↓
6. action.execute(parameters)
   ↓
7. Interact with Joomla APIs
   - Read: Query database, load models
   - Write: Update database, use models
   ↓
8. Return structured result
   ↓
9. Log action in database
   ↓
10. Add to conversation history
```

---

## Agent Reasoning Loop

The orchestrator implements a sophisticated reasoning loop:

### Initialization

```php
private function initializeConversation(string $userPrompt): void
{
    $systemPrompt = <<<SYSTEM
You are an AI assistant integrated into a Joomla 5 CMS...
Available actions:
- read_article: Get full article data
- search_articles: Search by keyword
- ...
SYSTEM;

    $this->conversationHistory = [
        ['role' => 'system', 'content' => $systemPrompt],
        ['role' => 'user', 'content' => $userPrompt]
    ];
}
```

### Iteration Loop

```php
while ($iteration < $this->maxIterations && !$isComplete) {
    // 1. Get AI's thought/action
    $response = $this->aiProvider->chat($this->conversationHistory);
    
    // 2. Parse actions
    $actions = $this->parseAgentResponse($response);
    
    // 3. Execute or queue
    foreach ($actions as $action) {
        if ($requireReview) {
            $action['status'] = 'pending_review';
        } else {
            $action = $this->executeAction($action);
            // 4. Feed results back
            $this->conversationHistory[] = [
                'role' => 'user',
                'content' => "Execution results: ..."
            ];
        }
    }
    
    // 5. Check completion
    if (stripos($response, 'COMPLETE') !== false) {
        $isComplete = true;
    }
}
```

### AI Response Format

The AI is instructed to respond in this JSON format:

```json
{
  "thought": "I need to first read the article, then update it",
  "actions": [
    {
      "type": "read_article",
      "parameters": {
        "article_id": 42
      },
      "reason": "Get current article content"
    },
    {
      "type": "update_article",
      "parameters": {
        "article_id": 42,
        "title": "New Title"
      },
      "reason": "Update the title as requested"
    }
  ]
}
```

---

## Action System

### Creating a New Action

```php
<?php
namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

class UpdateCategoryAction implements ActionInterface
{
    public function execute(array $parameters): array
    {
        $categoryId = $parameters['category_id'];
        
        // Load category
        $table = Table::getInstance('Category', 'Joomla\\CMS\\Table\\');
        $table->load($categoryId);
        
        // Update fields
        foreach ($parameters as $key => $value) {
            if (property_exists($table, $key)) {
                $table->$key = $value;
            }
        }
        
        // Save
        if (!$table->store()) {
            throw new \RuntimeException($table->getError());
        }
        
        return [
            'success' => true,
            'category_id' => $categoryId,
            'updated' => array_keys($parameters)
        ];
    }
    
    public static function getDescription(): string
    {
        return 'Update a category. Required: category_id. ' .
               'Optional: title, description, published, etc.';
    }
    
    public static function getRequiredParameters(): array
    {
        return ['category_id'];
    }
    
    public function validate(array $parameters): bool
    {
        return isset($parameters['category_id']) && 
               is_numeric($parameters['category_id']);
    }
}
```

### Registering the Action

In `ActionRegistry::registerDefaultActions()`:

```php
$this->register('update_category', Actions\UpdateCategoryAction::class);
```

### Permission Mapping

Add to permission map in `ActionRegistry::isActionAllowed()`:

```php
$permissionMap = [
    // ...
    'update_category' => 'write_categories',
];
```

Add to manifest XML:

```xml
<option value="write_categories">Write/Update Categories</option>
```

---

## Security Model

### Multi-Layer Security

1. **Authentication**
   - Only logged-in users
   - `core.admin` permission required
   - Checked on every request

2. **Authorization**
   - Fine-grained action permissions
   - Configured per-action type
   - Enforced by ActionRegistry

3. **Input Validation**
   - Each action validates parameters
   - Type checking
   - Required field verification

4. **Review Mode**
   - Optional manual approval
   - Preview all planned changes
   - Selective execution

5. **Audit Trail**
   - All actions logged
   - User attribution
   - Timestamp tracking
   - Result storage

### Permission Checks

```php
private function isActionAllowed(string $name): bool
{
    $allowedActions = $this->params['allowed_actions'] ?? [];
    $requiredPermission = $this->permissionMap[$name] ?? null;
    
    if (!$requiredPermission) {
        return false; // Deny unknown actions
    }
    
    return in_array($requiredPermission, $allowedActions);
}
```

### SQL Injection Prevention

All database queries use Joomla's query builder:

```php
$query = $db->getQuery(true)
    ->select('*')
    ->from($db->quoteName('#__content'))
    ->where($db->quoteName('id') . ' = ' . (int) $articleId);
```

---

## Extension Points

### 1. Custom AI Providers

Implement `AIProviderInterface`:

```php
class LocalLLMProvider implements AIProviderInterface
{
    public function chat(array $messages): string
    {
        // Call local model (e.g., Ollama, LM Studio)
        $response = $this->callLocalLLM($messages);
        return $response;
    }
}
```

Register in plugin configuration.

### 2. Custom Actions

Create action class → Register in ActionRegistry → Configure permissions

### 3. Custom UI

The console template can be completely replaced:
- Copy `tmpl/console.php`
- Modify HTML structure
- Update JavaScript if needed
- Maintain API contract

### 4. Event Hooks

The plugin could dispatch custom events:

```php
// In AgentOrchestrator
$this->dispatcher->dispatch('onBeforeActionExecute', $event);
$result = $action->execute($parameters);
$this->dispatcher->dispatch('onAfterActionExecute', $event);
```

Other extensions can listen and react.

### 5. Storage Backends

The ActionLogger could be abstracted:

```php
interface LoggerInterface
{
    public function startSession(string $prompt): string;
    public function logAction(string $sessionId, array $action): void;
    public function getRecentSessions(int $limit): array;
}
```

Implementations: DatabaseLogger, FileLogger, RemoteLogger

---

## Performance Considerations

### Optimization Strategies

1. **Lazy Loading**
   - Actions instantiated only when needed
   - AI provider created on demand
   - Models loaded per-request

2. **Caching**
   - Consider caching frequent reads
   - Site info could be cached
   - Template data cacheable

3. **Database Indexing**
   ```sql
   INDEX idx_user_started (user_id, started_at)
   INDEX idx_session (session_id)
   ```

4. **API Rate Limiting**
   - Implement in AI providers
   - Respect provider limits
   - Queue requests if needed

5. **Pagination**
   - Large result sets should paginate
   - Implement in search actions
   - Return counts separately

### Monitoring

Add instrumentation:

```php
private function executeAction(array $action): array
{
    $startTime = microtime(true);
    
    try {
        $result = $actionHandler->execute($parameters);
    } finally {
        $duration = microtime(true) - $startTime;
        $this->logger->logPerformance($action['type'], $duration);
    }
    
    return $result;
}
```

---

## Error Handling

### Graceful Degradation

```php
try {
    $result = $orchestrator->processPrompt($prompt);
} catch (AIProviderException $e) {
    // AI service down
    return ['error' => 'AI service unavailable', 'retry' => true];
} catch (ActionException $e) {
    // Action failed
    return ['error' => $e->getMessage(), 'action' => $e->getAction()];
} catch (\Exception $e) {
    // Unexpected error
    Log::add($e->getMessage(), Log::ERROR, 'aiassistant');
    return ['error' => 'An unexpected error occurred'];
}
```

### User-Friendly Messages

Transform technical errors:

```php
private function formatError(\Exception $e): string
{
    $technicalErrors = [
        'Connection refused' => 'Could not connect to AI service',
        'Invalid credentials' => 'API key is invalid or expired',
        'Rate limit exceeded' => 'Too many requests, please wait'
    ];
    
    foreach ($technicalErrors as $pattern => $friendly) {
        if (str_contains($e->getMessage(), $pattern)) {
            return $friendly;
        }
    }
    
    return 'An error occurred while processing your request';
}
```

---

## Testing Strategy

### Unit Tests

Test individual components:

```php
class ReadArticleActionTest extends TestCase
{
    public function testExecuteReturnsArticleData()
    {
        $action = new ReadArticleAction();
        $result = $action->execute(['article_id' => 1]);
        
        $this->assertArrayHasKey('title', $result);
        $this->assertArrayHasKey('introtext', $result);
    }
}
```

### Integration Tests

Test the full flow:

```php
public function testPromptToExecution()
{
    $orchestrator = $this->getOrchestrator();
    $result = $orchestrator->processPrompt(
        "Get information about article 1",
        false // Don't require review
    );
    
    $this->assertTrue($result['success']);
    $this->assertNotEmpty($result['actions']);
}
```

### Manual Testing Checklist

- [ ] Install plugin
- [ ] Configure API key
- [ ] Access console
- [ ] Submit read-only prompt
- [ ] Verify response
- [ ] Enable write permissions
- [ ] Test with review mode
- [ ] Execute reviewed action
- [ ] Check action logs
- [ ] Test error scenarios

---

## Future Enhancements

### Planned Features

1. **Streaming Responses**
   - Server-sent events for real-time AI thinking
   - Progressive action display

2. **Multi-User Sessions**
   - Collaborative editing
   - Session sharing

3. **Advanced Actions**
   - Template editing
   - Extension management
   - User management

4. **AI Memory**
   - Persistent context across sessions
   - Learn site structure
   - Remember preferences

5. **Natural Language Queries**
   - SQL generation from NL
   - Complex data analysis
   - Report generation

6. **Voice Interface**
   - Speech-to-text input
   - Text-to-speech output

7. **Mobile App**
   - Native iOS/Android
   - Push notifications
   - Offline queue

---

## Conclusion

The AI Assistant plugin demonstrates how modern AI can be integrated into CMS platforms to create powerful, intuitive content management tools. Its modular architecture makes it easy to extend while maintaining security and reliability.

The agentic approach—where the AI plans, executes, and verifies autonomously—provides a glimpse into the future of human-computer interaction for content management.

For questions or contributions, please refer to the main README or open an issue on GitHub.

