<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Query Database Action - Execute dynamic SELECT queries
 *
 * @since  1.0.0
 */
class QueryDatabaseAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Query results
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        Log::add('[QueryDatabase] Received parameters: ' . json_encode($parameters), Log::DEBUG, 'aiassistant');
        
        if (!$this->validate($parameters)) {
            Log::add('[QueryDatabase] Validation failed for parameters: ' . json_encode($parameters), Log::ERROR, 'aiassistant');
            throw new \InvalidArgumentException('Invalid parameters for QueryDatabaseAction');
        }

        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        $query = $parameters['query'];
        $limit = $parameters['limit'] ?? 100; // Default limit for safety
        
        Log::add('[QueryDatabase] Query before validation: ' . $query, Log::DEBUG, 'aiassistant');

        // Validate query is SELECT only
        if (!$this->isSelectQuery($query)) {
            Log::add('[QueryDatabase] REJECTED query: ' . $query, Log::ERROR, 'aiassistant');
            throw new \RuntimeException('Only SELECT queries are allowed. For write operations, use specific write actions. Rejected query: ' . substr($query, 0, 200));
        }
        
        // Replace #__ with actual database prefix
        $prefix = $db->getPrefix();
        $query = str_replace('#__', $prefix, $query);
        
        Log::add('[QueryDatabase] Query after prefix replacement: ' . $query, Log::DEBUG, 'aiassistant');

        // Add automatic LIMIT if not present
        if (stripos($query, 'LIMIT') === false) {
            $query .= ' LIMIT ' . (int) $limit;
        }

        Log::add('[QueryDatabase] Executing: ' . $query, Log::DEBUG, 'aiassistant');

        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();
            
            $resultCount = count($results);
            
            // If no results, provide helpful diagnostics
            $diagnostics = [];
            if ($resultCount === 0) {
                Log::add('[QueryDatabase] Query returned 0 results: ' . $query, Log::WARNING, 'aiassistant');
                
                // Try to extract table name from query
                if (preg_match('/FROM\s+([`#\w]+)/i', $query, $matches)) {
                    $tableName = $matches[1];
                    
                    try {
                        // Check if table exists and has any rows
                        $db->setQuery("SELECT COUNT(*) FROM {$tableName}");
                        $totalRows = $db->loadResult();
                        
                        $diagnostics['table_total_rows'] = $totalRows;
                        $diagnostics['message'] = $totalRows > 0 
                            ? "Table {$tableName} has {$totalRows} total rows, but 0 matched your query. Check your WHERE conditions."
                            : "Table {$tableName} exists but is completely empty (0 rows total).";
                        
                        Log::add('[QueryDatabase] Diagnostic: ' . $diagnostics['message'], Log::INFO, 'aiassistant');
                    } catch (\Exception $diagEx) {
                        $diagnostics['message'] = "Could not check table status: " . $diagEx->getMessage();
                    }
                }
            }

            $result = [
                'success' => true,
                'query' => $query,
                'count' => $resultCount,
                'results' => $results
            ];
            
            if (!empty($diagnostics)) {
                $result['diagnostics'] = $diagnostics;
            }
            
            return $result;
        } catch (\Exception $e) {
            Log::add('[QueryDatabase] Error: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
            
            $errorMsg = $e->getMessage();
            $hint = '';
            
            // Provide helpful hints for common errors
            if (stripos($errorMsg, 'Unknown column') !== false) {
                // Extract table name from query
                if (preg_match('/FROM\s+([`#\w]+)/i', $query, $matches)) {
                    $tableName = str_replace($prefix, '#__', $matches[1]);
                    $tableName = str_replace('`', '', $tableName);
                    $hint = "\n\n💡 TIP: The column you specified doesn't exist. Use describe_table action on '{$tableName}' to see available columns, then retry with correct column names!";
                }
            } elseif (stripos($errorMsg, "doesn't exist") !== false && stripos($errorMsg, 'Table') !== false) {
                $hint = "\n\n💡 TIP: This table doesn't exist. Use list_tables to see all available tables.";
            } elseif (stripos($errorMsg, 'syntax error') !== false) {
                $hint = "\n\n💡 TIP: SQL syntax error. Check for reserved words (`fulltext`, `order`, `key`) and use backticks around them.";
            }
            
            return [
                'success' => false,
                'error' => $errorMsg . $hint,
                'query' => $query
            ];
        }
    }

    /**
     * Validate that query is a safe SELECT statement
     *
     * @param   string  $query  SQL query
     *
     * @return  boolean
     * @since   1.0.0
     */
    private function isSelectQuery(string $query): bool
    {
        // Clean up query - remove extra whitespace, newlines, etc.
        $query = preg_replace('/\s+/', ' ', trim($query));
        $queryUpper = strtoupper($query);
        
        // Must start with SELECT (allowing whitespace)
        if (!preg_match('/^\s*SELECT\s+/i', $query)) {
            Log::add('[QueryDatabase] Query does not start with SELECT: ' . substr($query, 0, 50), Log::WARNING, 'aiassistant');
            return false;
        }

        // Must not contain dangerous keywords
        $dangerousKeywords = [
            'INSERT', 'UPDATE', 'DELETE', 'DROP', 'CREATE', 'ALTER',
            'TRUNCATE', 'GRANT', 'REVOKE', 'EXEC', 'EXECUTE'
        ];

        foreach ($dangerousKeywords as $keyword) {
            // Use word boundaries to avoid false positives (e.g., "SELECTED" shouldn't trigger "DELETE")
            if (preg_match('/\b' . preg_quote($keyword, '/') . '\b/i', $queryUpper)) {
                Log::add('[QueryDatabase] Query contains dangerous keyword ' . $keyword . ': ' . substr($query, 0, 100), Log::WARNING, 'aiassistant');
                return false;
            }
        }

        return true;
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Execute a dynamic SELECT query on the Joomla database. Use this to query ANY table. ' .
               'Parameters: query (SELECT statement), limit (optional, max 1000). ' .
               'Example: SELECT * FROM #__content WHERE state=1. ' .
               'Common tables: #__content (articles), #__modules, #__menu, #__categories, #__extensions, #__sppagebuilder_pages';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['query'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return !empty($parameters['query']) && is_string($parameters['query']);
    }
}

