<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Query Database Action - Execute dynamic SELECT queries
 *
 * @since  1.0.0
 */
class QueryDatabaseAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Query results
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for QueryDatabaseAction');
        }

        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        $query = $parameters['query'];
        $limit = $parameters['limit'] ?? 100; // Default limit for safety

        // Validate query is SELECT only
        if (!$this->isSelectQuery($query)) {
            Log::add('[QueryDatabase] REJECTED query: ' . $query, Log::ERROR, 'aiassistant');
            throw new \RuntimeException('Only SELECT queries are allowed. For write operations, use specific write actions. Rejected query: ' . substr($query, 0, 200));
        }

        // Add automatic LIMIT if not present
        if (stripos($query, 'LIMIT') === false) {
            $query .= ' LIMIT ' . (int) $limit;
        }

        Log::add('[QueryDatabase] Executing: ' . $query, Log::DEBUG, 'aiassistant');

        try {
            $db->setQuery($query);
            $results = $db->loadObjectList();

            return [
                'success' => true,
                'query' => $query,
                'count' => count($results),
                'results' => $results
            ];
        } catch (\Exception $e) {
            Log::add('[QueryDatabase] Error: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'query' => $query
            ];
        }
    }

    /**
     * Validate that query is a safe SELECT statement
     *
     * @param   string  $query  SQL query
     *
     * @return  boolean
     * @since   1.0.0
     */
    private function isSelectQuery(string $query): bool
    {
        $query = trim(strtoupper($query));
        
        // Must start with SELECT
        if (strpos($query, 'SELECT') !== 0) {
            return false;
        }

        // Must not contain dangerous keywords
        $dangerousKeywords = [
            'INSERT', 'UPDATE', 'DELETE', 'DROP', 'CREATE', 'ALTER',
            'TRUNCATE', 'GRANT', 'REVOKE', 'EXEC', 'EXECUTE'
        ];

        foreach ($dangerousKeywords as $keyword) {
            if (strpos($query, $keyword) !== false) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Execute a dynamic SELECT query on the Joomla database. Use this to query ANY table. ' .
               'Parameters: query (SELECT statement), limit (optional, max 1000). ' .
               'Example: SELECT * FROM #__content WHERE state=1. ' .
               'Common tables: #__content (articles), #__modules, #__menu, #__categories, #__extensions, #__sppagebuilder_pages';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['query'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return !empty($parameters['query']) && is_string($parameters['query']);
    }
}

