const { Storage } = require('@google-cloud/storage');
const path = require('path');
const fs = require('fs');
const sharp = require('sharp');

// Initialize Cloud Storage client
// When running on Cloud Run, credentials are automatically provided
const storage = new Storage();

const BUCKET_NAME = process.env.GCS_BUCKET_NAME;
const USE_CLOUD_STORAGE = process.env.USE_CLOUD_STORAGE === 'true';

/**
 * Upload a file to Cloud Storage
 * @param {string} localFilePath - Local file path
 * @param {string} destinationPath - Destination path in bucket (e.g., "1/image.jpg")
 * @returns {Promise<string>} - Public URL of the uploaded file
 */
async function uploadFile(localFilePath, destinationPath) {
  if (!USE_CLOUD_STORAGE) {
    // Return local path for development
    return `/uploads/${destinationPath}`;
  }

  const bucket = storage.bucket(BUCKET_NAME);
  const destination = `uploads/${destinationPath}`;
  
  await bucket.upload(localFilePath, {
    destination: destination,
    metadata: {
      cacheControl: 'public, max-age=31536000', // Cache for 1 year
    },
  });

  // Return the public URL
  return `https://storage.googleapis.com/${BUCKET_NAME}/${destination}`;
}

/**
 * Upload a buffer to Cloud Storage
 * @param {Buffer} buffer - File buffer
 * @param {string} destinationPath - Destination path in bucket
 * @param {string} contentType - MIME type
 * @returns {Promise<string>} - Public URL of the uploaded file
 */
async function uploadBuffer(buffer, destinationPath, contentType) {
  if (!USE_CLOUD_STORAGE) {
    // For development, save to local filesystem
    const localPath = path.join(__dirname, '..', 'uploads', destinationPath);
    const dir = path.dirname(localPath);
    if (!fs.existsSync(dir)) {
      fs.mkdirSync(dir, { recursive: true });
    }
    fs.writeFileSync(localPath, buffer);
    return `/uploads/${destinationPath}`;
  }

  const bucket = storage.bucket(BUCKET_NAME);
  const destination = `uploads/${destinationPath}`;
  const file = bucket.file(destination);

  await file.save(buffer, {
    metadata: {
      contentType: contentType,
      cacheControl: 'public, max-age=31536000',
    },
  });

  return `https://storage.googleapis.com/${BUCKET_NAME}/${destination}`;
}

/**
 * Generate and upload thumbnail
 * @param {string} sourcePathOrUrl - Source image path or URL
 * @param {number} albumId - Album ID
 * @param {string} filename - Filename
 * @returns {Promise<string>} - URL of the thumbnail
 */
async function generateAndUploadThumbnail(sourcePathOrUrl, albumId, filename) {
  try {
    let imageBuffer;
    
    if (USE_CLOUD_STORAGE) {
      // Download from Cloud Storage
      const bucket = storage.bucket(BUCKET_NAME);
      const file = bucket.file(`uploads/${albumId}/${filename}`);
      const [buffer] = await file.download();
      imageBuffer = buffer;
    } else {
      // Read from local filesystem
      const localPath = path.join(__dirname, '..', 'uploads', String(albumId), filename);
      imageBuffer = fs.readFileSync(localPath);
    }

    // Generate thumbnail
    const thumbnailBuffer = await sharp(imageBuffer)
      .resize(600, null, {
        withoutEnlargement: true,
        fit: 'inside'
      })
      .jpeg({ quality: 92, progressive: true })
      .toBuffer();

    // Upload thumbnail
    const thumbPath = `${albumId}/thumbs/${filename}`;
    const thumbUrl = await uploadBuffer(thumbnailBuffer, thumbPath, 'image/jpeg');
    
    return thumbUrl;
  } catch (error) {
    console.error('Error generating thumbnail:', error);
    throw error;
  }
}

/**
 * Check if a file exists in Cloud Storage
 * @param {string} filePath - File path in bucket (e.g., "uploads/1/image.jpg")
 * @returns {Promise<boolean>}
 */
async function fileExists(filePath) {
  if (!USE_CLOUD_STORAGE) {
    const localPath = path.join(__dirname, '..', 'uploads', filePath);
    return fs.existsSync(localPath);
  }

  try {
    const bucket = storage.bucket(BUCKET_NAME);
    const file = bucket.file(`uploads/${filePath}`);
    const [exists] = await file.exists();
    return exists;
  } catch (error) {
    return false;
  }
}

/**
 * Get a signed URL for a file (for private access)
 * @param {string} filePath - File path in bucket
 * @param {number} expiresInMinutes - URL expiration time
 * @returns {Promise<string>}
 */
async function getSignedUrl(filePath, expiresInMinutes = 60) {
  if (!USE_CLOUD_STORAGE) {
    return `/uploads/${filePath}`;
  }

  const bucket = storage.bucket(BUCKET_NAME);
  const file = bucket.file(`uploads/${filePath}`);

  const [url] = await file.getSignedUrl({
    action: 'read',
    expires: Date.now() + expiresInMinutes * 60 * 1000,
  });

  return url;
}

/**
 * Delete a file from Cloud Storage
 * @param {string} filePath - File path in bucket
 * @returns {Promise<void>}
 */
async function deleteFile(filePath) {
  if (!USE_CLOUD_STORAGE) {
    const localPath = path.join(__dirname, '..', 'uploads', filePath);
    if (fs.existsSync(localPath)) {
      fs.unlinkSync(localPath);
    }
    return;
  }

  const bucket = storage.bucket(BUCKET_NAME);
  const file = bucket.file(`uploads/${filePath}`);
  await file.delete();
}

/**
 * Get public URL for a file
 * @param {string} albumId - Album ID
 * @param {string} filename - Filename
 * @param {boolean} isThumbnail - Whether it's a thumbnail
 * @returns {string}
 */
function getPublicUrl(albumId, filename, isThumbnail = false) {
  if (!USE_CLOUD_STORAGE) {
    if (isThumbnail) {
      return `/thumb/${albumId}/${filename}`;
    }
    return `/uploads/${albumId}/${filename}`;
  }

  const path = isThumbnail 
    ? `uploads/${albumId}/thumbs/${filename}`
    : `uploads/${albumId}/${filename}`;
  
  return `https://storage.googleapis.com/${BUCKET_NAME}/${path}`;
}

module.exports = {
  uploadFile,
  uploadBuffer,
  generateAndUploadThumbnail,
  fileExists,
  getSignedUrl,
  deleteFile,
  getPublicUrl,
  USE_CLOUD_STORAGE,
};

