<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Registry\Registry;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Update Module Action
 *
 * @since  1.0.0
 */
class UpdateModuleAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Update result
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for UpdateModuleAction');
        }

        try {
            $moduleId = $parameters['module_id'];
            $table = Table::getInstance('Module', 'Joomla\\CMS\\Table\\');

            if (!$table) {
                throw new \RuntimeException('Failed to load module table');
            }

            if (!$table->load($moduleId)) {
                throw new \RuntimeException("Module with ID {$moduleId} not found");
            }

            // Update basic fields
            $allowedFields = ['title', 'content', 'position', 'published', 'ordering', 'note'];
            foreach ($allowedFields as $field) {
                if (isset($parameters[$field])) {
                    $table->$field = $parameters[$field];
                }
            }

            // Update module parameters
            if (isset($parameters['params'])) {
                $currentParams = new Registry($table->params);
                
                if (is_array($parameters['params'])) {
                    foreach ($parameters['params'] as $key => $value) {
                        $currentParams->set($key, $value);
                    }
                }
                
                $table->params = $currentParams->toString();
            }

            if (!$table->store()) {
                throw new \RuntimeException('Failed to update module: ' . $table->getError());
            }

            return [
                'success' => true,
                'module_id' => $moduleId,
                'updated_fields' => array_keys(array_intersect_key($parameters, array_flip($allowedFields))),
                'message' => 'Module updated successfully'
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to update module: ' . $e->getMessage());
        }
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Update a module. Required: module_id. Optional: title, content, position, published, ordering, note, params (object).';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['module_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['module_id']) && is_numeric($parameters['module_id']);
    }
}

