<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Table\Table;
use Joomla\Registry\Registry;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Undo Action - Reverts a previous action
 *
 * @since  1.0.0
 */
class UndoActionAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Undo result
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for UndoActionAction');
        }

        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            $actionId = $parameters['action_id'];

            // Get the original action details
            $query = $db->getQuery(true)
                ->select('*')
                ->from($db->quoteName('#__aiassistant_actions'))
                ->where($db->quoteName('id') . ' = ' . (int) $actionId);

            $db->setQuery($query);
            $action = $db->loadObject();

            if (!$action) {
                throw new \RuntimeException("Action with ID {$actionId} not found");
            }

            if ($action->status !== 'executed') {
                throw new \RuntimeException("Cannot undo action that was not executed");
            }

            $beforeState = json_decode($action->before_state, true);
            if (!$beforeState) {
                throw new \RuntimeException("No previous state available for this action");
            }

            // Revert based on action type
            $revertResult = $this->revertAction($action->action_type, $beforeState);

            // Mark action as undone
            $query = $db->getQuery(true)
                ->update($db->quoteName('#__aiassistant_actions'))
                ->set($db->quoteName('status') . ' = ' . $db->quote('undone'))
                ->set($db->quoteName('undone_at') . ' = ' . $db->quote(Factory::getDate()->toSql()))
                ->where($db->quoteName('id') . ' = ' . (int) $actionId);

            $db->setQuery($query);
            $db->execute();

            return [
                'success' => true,
                'action_id' => $actionId,
                'action_type' => $action->action_type,
                'reverted' => $revertResult,
                'message' => 'Action successfully undone'
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to undo action: ' . $e->getMessage());
        }
    }

    /**
     * Revert an action based on its type
     *
     * @param   string  $actionType   Type of action to revert
     * @param   array   $beforeState  Previous state
     *
     * @return  array   Revert result
     * @since   1.0.0
     */
    private function revertAction(string $actionType, array $beforeState): array
    {
        switch ($actionType) {
            case 'create_article':
                return $this->revertCreateArticle($beforeState);

            case 'update_article':
                return $this->revertUpdateArticle($beforeState);

            case 'update_module':
                return $this->revertUpdateModule($beforeState);

            case 'update_pagebuilder_page':
                return $this->revertUpdatePageBuilder($beforeState);

            default:
                throw new \RuntimeException("Undo not supported for action type: {$actionType}");
        }
    }

    /**
     * Revert article creation by deleting it
     *
     * @param   array  $beforeState  Previous state
     *
     * @return  array  Result
     * @since   1.0.0
     */
    private function revertCreateArticle(array $beforeState): array
    {
        if (empty($beforeState['article_id'])) {
            throw new \RuntimeException('Article ID not found in previous state');
        }

        BaseDatabaseModel::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_content/models');
        $model = BaseDatabaseModel::getInstance('Article', 'ContentModel');

        if (!$model) {
            throw new \RuntimeException('Failed to load article model');
        }

        // Delete the article (send to trash)
        if (!$model->delete([$beforeState['article_id']])) {
            throw new \RuntimeException('Failed to delete article: ' . $model->getError());
        }

        return [
            'article_id' => $beforeState['article_id'],
            'action' => 'deleted'
        ];
    }

    /**
     * Revert article update by restoring previous values
     *
     * @param   array  $beforeState  Previous state
     *
     * @return  array  Result
     * @since   1.0.0
     */
    private function revertUpdateArticle(array $beforeState): array
    {
        if (empty($beforeState['id'])) {
            throw new \RuntimeException('Article ID not found in previous state');
        }

        BaseDatabaseModel::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_content/models');
        $model = BaseDatabaseModel::getInstance('Article', 'ContentModel');

        if (!$model) {
            throw new \RuntimeException('Failed to load article model');
        }

        // Restore all previous values
        if (!$model->save($beforeState)) {
            throw new \RuntimeException('Failed to restore article: ' . $model->getError());
        }

        return [
            'article_id' => $beforeState['id'],
            'action' => 'restored',
            'fields_restored' => array_keys($beforeState)
        ];
    }

    /**
     * Revert module update by restoring previous values
     *
     * @param   array  $beforeState  Previous state
     *
     * @return  array  Result
     * @since   1.0.0
     */
    private function revertUpdateModule(array $beforeState): array
    {
        if (empty($beforeState['id'])) {
            throw new \RuntimeException('Module ID not found in previous state');
        }

        $table = Table::getInstance('Module', 'Joomla\\CMS\\Table\\');

        if (!$table) {
            throw new \RuntimeException('Failed to load module table');
        }

        if (!$table->bind($beforeState)) {
            throw new \RuntimeException('Failed to bind data to module table');
        }

        if (!$table->store()) {
            throw new \RuntimeException('Failed to restore module: ' . $table->getError());
        }

        return [
            'module_id' => $beforeState['id'],
            'action' => 'restored',
            'fields_restored' => array_keys($beforeState)
        ];
    }

    /**
     * Revert page builder update by restoring previous values
     *
     * @param   array  $beforeState  Previous state
     *
     * @return  array  Result
     * @since   1.0.0
     */
    private function revertUpdatePageBuilder(array $beforeState): array
    {
        if (empty($beforeState['id'])) {
            throw new \RuntimeException('Page ID not found in previous state');
        }

        $db = Factory::getContainer()->get('DatabaseDriver');

        $query = $db->getQuery(true)
            ->update($db->quoteName('#__sppagebuilder'))
            ->set($db->quoteName('text') . ' = ' . $db->quote($beforeState['text'] ?? ''))
            ->set($db->quoteName('title') . ' = ' . $db->quote($beforeState['title'] ?? ''))
            ->where($db->quoteName('id') . ' = ' . (int) $beforeState['id']);

        $db->setQuery($query);
        $db->execute();

        return [
            'page_id' => $beforeState['id'],
            'action' => 'restored',
            'fields_restored' => array_keys($beforeState)
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Undo a previously executed action. Required: action_id (the ID of the action to undo). This will restore the content to its state before the action was performed.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['action_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['action_id']) && is_numeric($parameters['action_id']);
    }
}

