<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Update Article Action
 *
 * @since  1.0.0
 */
class UpdateArticleAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Update result
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for UpdateArticleAction');
        }

        $articleId = $parameters['article_id'];
        unset($parameters['article_id']);

        try {
            // Load the article model
            BaseDatabaseModel::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_content/models');
            $model = BaseDatabaseModel::getInstance('Article', 'ContentModel');

            if (!$model) {
                throw new \RuntimeException('Failed to load article model');
            }

            // Get current article data
            $article = $model->getItem($articleId);

            if (!$article || !$article->id) {
                throw new \RuntimeException("Article with ID {$articleId} not found");
            }

            // Prepare data for update
            $data = array_merge((array) $article, $parameters);
            $data['id'] = $articleId;

            // Perform the update
            if (!$model->save($data)) {
                throw new \RuntimeException('Failed to update article: ' . $model->getError());
            }

            return [
                'success' => true,
                'article_id' => $articleId,
                'updated_fields' => array_keys($parameters),
                'message' => 'Article updated successfully'
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to update article: ' . $e->getMessage());
        }
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Update an existing article. Required: article_id. Optional fields: title, alias, introtext, fulltext, state, catid, metadesc, metakey, featured, etc.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['article_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['article_id']) && is_numeric($parameters['article_id']);
    }
}

