# AI Assistant Plugin - Dynamic Branding Implementation

## Overview

All branding elements (colors, logos, text, titles) in the AI Assistant plugin are now **dynamically controlled** via the Joomla administrator plugin settings. No branding is hardcoded in the CSS, JavaScript, or HTML.

## What Was Changed

### 1. Admin Configuration Settings (aiassistant.xml)

Added a new **"Branding & Appearance"** fieldset with the following configurable options:

#### Text & Content
- **Console Title** - The main title displayed in the console header
- **Welcome Heading** - The heading shown on the welcome screen
- **Welcome Text** - The descriptive text shown on the welcome screen
- **Custom Logo** - Upload a custom logo (supports all image formats via Joomla media manager)

#### Color Scheme (13 customizable colors)
- **Primary Color** - Main accent color for buttons and highlights
- **Primary Hover Color** - Color when hovering over primary elements
- **Secondary Color** - Secondary accent color for less prominent elements
- **Success Color** - Color for success messages and completed actions
- **Warning Color** - Color for warnings and pending actions
- **Error Color** - Color for error messages and failed actions
- **Background Color** - Main background color for panels
- **Alternate Background Color** - Alternate background for contrast
- **Border Color** - Color for borders and dividers
- **Text Color** - Main text color
- **Muted Text Color** - Color for secondary text
- **Thinking Gradient Start** - Start color for the thinking animation gradient
- **Thinking Gradient End** - End color for the thinking animation gradient

### 2. Template File (tmpl/console.php)

**Dynamic Loading of Settings:**
```php
// Loads plugin parameters from database
$db = Factory::getDbo();
$params = new \Joomla\Registry\Registry($paramsJson);

// Gets branding settings with fallback defaults
$consoleTitle = $params->get('console_title', 'AI Assistant Console');
$welcomeHeading = $params->get('console_welcome_heading', 'Welcome to AI Assistant');
$customLogo = $params->get('custom_logo', '');
$primaryColor = $params->get('primary_color', '#3b82f6');
// ... etc
```

**Dynamic CSS Variables:**
- CSS variables are now set dynamically in an inline `<style>` block
- All colors are injected from admin settings
- Falls back to sensible defaults if not configured

**Dynamic Logo:**
- Shows custom uploaded logo if configured
- Falls back to default SVG icon if no logo is uploaded

**Dynamic Text:**
- Console title, welcome heading, and welcome text all use admin settings
- Properly escaped with `htmlspecialchars()` for security

**Configuration Passed to JavaScript:**
```javascript
window.AI_ASSISTANT_CONFIG = {
    baseUrl: '...',
    token: '...',
    branding: {
        title: "...",
        welcomeHeading: "...",
        welcomeText: "...",
        logo: "...",
        colors: { ... }
    }
};
```

### 3. CSS File (media/css/console.css)

**All Hardcoded Colors Removed:**
- Replaced all hardcoded color values with CSS variables
- Example: Changed `color: #3b82f6` to `color: var(--ai-primary)`
- Changed `background: #f0fdf4` to `background: color-mix(in srgb, var(--ai-success) 10%, white)`

**Areas Updated:**
- Message backgrounds and text colors
- Button colors and hover states
- Status indicators (success/warning/error)
- Thinking animation gradient
- Border and shadow colors
- All text colors (normal and muted)
- Action item backgrounds
- Modal and sidebar colors

**CSS Variables Used:**
- `--ai-primary`
- `--ai-primary-hover`
- `--ai-secondary`
- `--ai-success`
- `--ai-warning`
- `--ai-error`
- `--ai-bg`
- `--ai-bg-alt`
- `--ai-border`
- `--ai-text`
- `--ai-text-muted`
- `--ai-thinking-gradient-start`
- `--ai-thinking-gradient-end`
- `--ai-shadow`
- `--ai-shadow-lg`

### 4. JavaScript File (media/js/console.js)

**Dynamic Branding Access:**
```javascript
const branding = window.AI_ASSISTANT_CONFIG.branding || {};
const colors = branding.colors || {};
const welcomeHeading = branding.welcomeHeading || 'Welcome to AI Assistant';
```

**Functions Updated to Use Dynamic Colors:**

1. **`clearConversation()`** - Uses dynamic welcome heading and text
2. **`testApiConnection()`** - Uses dynamic warning and success colors for diagnostic messages
3. **`viewLogs()`** - Uses dynamic colors for log level highlighting (error/warning/success)

**Examples:**
```javascript
// Dynamic welcome screen
conversation.innerHTML = `
    <div class="ai-welcome">
        <h2>${this.escapeHtml(welcomeHeading)}</h2>
        <p>${this.escapeHtml(welcomeText)}</p>
        ...
    </div>
`;

// Dynamic log colors
let color = successColor;
if (log.includes('[ERROR]') || log.includes('[FATAL]')) {
    color = errorColor;
} else if (log.includes('[WARN]')) {
    color = warningColor;
}
```

### 5. Language Strings (language/en-GB/plg_system_aiassistant.ini)

Added all necessary language strings for the new branding settings:
- Field labels and descriptions for all 13+ branding options
- Organized under "; Branding" section
- Follows Joomla naming conventions

## How Admins Customize Branding

1. **Navigate to Plugin Settings:**
   - Go to: System → Plugins → System - AI Assistant
   - Or: Extensions → Plugins → Filter by "AI Assistant"

2. **Configure Branding Tab:**
   - Click on the "Branding & Appearance" fieldset
   - Modify any text fields (title, headings, welcome message)
   - Upload a custom logo if desired
   - Change any of the 13 color settings using color pickers

3. **Save Changes:**
   - Click "Save" or "Save & Close"
   - Changes take effect immediately on next console page load

4. **Reset to Defaults:**
   - Clear any field to use the default value
   - Default values are sensible, professional blue theme

## Benefits of This Implementation

✅ **No Hardcoded Branding** - Everything is configurable
✅ **Centralized Management** - All settings in one place
✅ **Real-time Updates** - Changes apply immediately
✅ **Security** - All output is properly escaped
✅ **Fallback Defaults** - Works even with empty settings
✅ **Professional UI** - Uses Joomla's native color picker and media manager
✅ **Maintainable** - CSS variables make it easy to extend
✅ **White-label Ready** - Agencies can rebrand for clients
✅ **Consistent** - JavaScript dynamically uses same colors as CSS

## Technical Details

### Security Considerations
- All dynamic text is escaped with `htmlspecialchars()` in PHP
- All dynamic text is escaped with `escapeHtml()` in JavaScript
- JSON encoding is used for passing data to JavaScript
- No user input directly rendered without sanitization

### Performance
- Colors are loaded once from database at page load
- CSS variables enable instant theme changes without page reload
- No additional HTTP requests for branding data

### Browser Compatibility
- Uses modern CSS variables (supported in all modern browsers)
- Uses `color-mix()` for dynamic color tinting (graceful fallback)
- SVG icon has fallback to custom logo

### Future Enhancements
Potential additions:
- Dark mode toggle
- Multiple saved color presets
- Import/export branding settings
- Per-user branding preferences
- Custom CSS injection field

## Files Modified

1. ✅ `plugins/system/aiassistant/aiassistant.xml` - Added branding fieldset
2. ✅ `plugins/system/aiassistant/language/en-GB/plg_system_aiassistant.ini` - Added language strings
3. ✅ `plugins/system/aiassistant/tmpl/console.php` - Dynamic branding loading and rendering
4. ✅ `plugins/system/aiassistant/media/css/console.css` - Removed hardcoded colors, use CSS variables
5. ✅ `plugins/system/aiassistant/media/js/console.js` - Dynamic branding in JavaScript

## Testing Checklist

- [ ] Change console title and verify it appears in header
- [ ] Change welcome heading/text and verify on welcome screen
- [ ] Upload custom logo and verify it replaces default icon
- [ ] Change primary color and verify buttons update
- [ ] Change error color and verify error messages update
- [ ] Change success color and verify completed actions update
- [ ] Change thinking gradient and verify animation colors
- [ ] Test with empty settings (should use defaults)
- [ ] Test log viewer colors
- [ ] Test API diagnostic colors
- [ ] Verify all text is properly escaped (no XSS vulnerabilities)
- [ ] Test on mobile devices
- [ ] Test in different browsers

## Conclusion

The AI Assistant plugin now has **complete dynamic branding support**. Every visual element, from colors to logos to text, can be customized by the site administrator without touching any code. This makes the plugin suitable for white-labeling, client customization, and brand consistency.

