<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Describe Table Action - Show table structure/schema
 *
 * @since  1.0.0
 */
class DescribeTableAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Table structure
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for DescribeTableAction');
        }

        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        $tableName = $parameters['table'];
        
        // Replace #__ with actual prefix
        $prefix = $db->getPrefix();
        $fullTableName = str_replace('#__', $prefix, $tableName);

        // Get table columns
        $columns = $db->getTableColumns($fullTableName, false);

        // Get sample row count
        $query = $db->getQuery(true)
            ->select('COUNT(*)')
            ->from($db->quoteName($fullTableName));
        
        try {
            $db->setQuery($query);
            $rowCount = $db->loadResult();
        } catch (\Exception $e) {
            $rowCount = 0;
        }

        return [
            'success' => true,
            'table' => $tableName,
            'full_table_name' => $fullTableName,
            'row_count' => $rowCount,
            'columns' => $columns
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Describe a database table structure. Shows columns, types, and row count. ' .
               'Parameter: table (use #__ prefix, e.g., "#__content"). ' .
               'Use this to understand what data a table contains before querying it.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['table'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return !empty($parameters['table']) && is_string($parameters['table']);
    }
}


