# Multi-Site Deployment Guide

This guide explains how to deploy this photo website to multiple Cloud Run instances with different configurations.

## Setup

### 1. Initialize Git Repository (First Time Only)

```bash
git init
git add .
git commit -m "Initial commit"
```

### 2. Push to GitHub

```bash
# Create a new repository on GitHub, then:
git remote add origin https://github.com/YOUR_USERNAME/photowebsite.git
git branch -M main
git push -u origin main
```

### 3. Deploy a New Site Instance

For each site, you'll need:
1. A new GCP project (or use the same project with different service names)
2. A Cloud Storage bucket for photos and data
3. Environment variables configured via Secret Manager

#### Create Cloud Storage Bucket

```bash
PROJECT_ID=your-project-id
SITE_NAME=site1  # Change for each site
BUCKET_NAME=${PROJECT_ID}-${SITE_NAME}

gsutil mb -l europe-west1 gs://${BUCKET_NAME}
gsutil iam ch allUsers:objectViewer gs://${BUCKET_NAME}
```

#### Set Environment Variables

```bash
# Set admin password
echo -n "your-secret-password" | gcloud secrets create ADMIN_PASSWORD --data-file=- --project=${PROJECT_ID}

# Site-specific configuration
gcloud run deploy photowebsite-${SITE_NAME} \
  --source . \
  --platform managed \
  --region europe-west1 \
  --allow-unauthenticated \
  --set-env-vars="USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${BUCKET_NAME}" \
  --set-env-vars="AGE_VERIFICATION_ENABLED=true,PRIMARY_COLOR=#a6ff00,LOGO_URL=/logo.png,SITE_NAME=My Site" \
  --set-secrets="ADMIN_PASSWORD=ADMIN_PASSWORD:latest" \
  --project=${PROJECT_ID}
```

## Configuration Options

### Environment Variables

| Variable | Description | Default |
|----------|-------------|---------|
| `ADMIN_PASSWORD` | Admin panel password | `changeme` |
| `AGE_VERIFICATION_ENABLED` | Enable 18+ age gate | `true` |
| `PRIMARY_COLOR` | Hex color for highlights | `#a6ff00` |
| `LOGO_URL` | Path or URL to logo | `/logo.png` |
| `SITE_NAME` | Site name for branding | `Lucky Pup Photography` |
| `USE_JSON` | Use JSON database | `true` |
| `USE_CLOUD_STORAGE` | Use GCS for photos | `true` |
| `GCS_BUCKET_NAME` | GCS bucket name | (required) |

### Updating an Existing Site

```bash
# Pull latest code
git pull origin main

# Redeploy with updated code
gcloud run deploy photowebsite-${SITE_NAME} \
  --source . \
  --platform managed \
  --region europe-west1 \
  --project=${PROJECT_ID}
```

### Changing Configuration

```bash
# Update environment variables only (no redeploy needed for these)
gcloud run services update photowebsite-${SITE_NAME} \
  --update-env-vars="PRIMARY_COLOR=#ff6b00,SITE_NAME=New Site Name" \
  --region europe-west1 \
  --project=${PROJECT_ID}
```

## Custom Logo

Upload your logo to the `public/` directory and reference it:

```bash
# Either commit to repo:
cp /path/to/logo.png public/
git add public/logo.png
git commit -m "Add custom logo"
git push

# Or upload directly to Cloud Storage and use URL:
gsutil cp /path/to/logo.png gs://${BUCKET_NAME}/
# Then set LOGO_URL=https://storage.googleapis.com/${BUCKET_NAME}/logo.png
```

## Automated Deployment with Cloud Build

Create `cloudbuild-site1.yaml`:

```yaml
steps:
  - name: 'gcr.io/cloud-builders/docker'
    args: ['build', '-t', 'gcr.io/$PROJECT_ID/photowebsite-site1:$COMMIT_SHA', '.']
  - name: 'gcr.io/cloud-builders/docker'
    args: ['push', 'gcr.io/$PROJECT_ID/photowebsite-site1:$COMMIT_SHA']
  - name: 'gcr.io/google.com/cloudsdktool/cloud-sdk'
    entrypoint: gcloud
    args:
      - 'run'
      - 'deploy'
      - 'photowebsite-site1'
      - '--image=gcr.io/$PROJECT_ID/photowebsite-site1:$COMMIT_SHA'
      - '--region=europe-west1'
      - '--platform=managed'
      - '--allow-unauthenticated'
      - '--set-env-vars=USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${_GCS_BUCKET}'
      - '--set-env-vars=AGE_VERIFICATION_ENABLED=${_AGE_VERIFICATION},PRIMARY_COLOR=${_PRIMARY_COLOR},LOGO_URL=${_LOGO_URL},SITE_NAME=${_SITE_NAME}'
      - '--set-secrets=ADMIN_PASSWORD=ADMIN_PASSWORD:latest'
images:
  - 'gcr.io/$PROJECT_ID/photowebsite-site1:$COMMIT_SHA'
```

Then deploy with:

```bash
gcloud builds submit --config=cloudbuild-site1.yaml \
  --substitutions=_GCS_BUCKET=bucket-name,_AGE_VERIFICATION=true,_PRIMARY_COLOR=#a6ff00,_LOGO_URL=/logo.png,_SITE_NAME="Site Name"
```
