# Google Cloud Run Deployment Guide

This guide will walk you through deploying the Lucky Pup Photography website to Google Cloud Run.

## Prerequisites

1. **Google Cloud Platform Account** with billing enabled
2. **gcloud CLI** installed and configured
   ```bash
   # Install: https://cloud.google.com/sdk/docs/install
   gcloud auth login
   gcloud config set project YOUR_PROJECT_ID
   ```
3. **Enabled APIs**
   - Cloud Run API
   - Cloud Build API
   - Cloud SQL Admin API
   - Secret Manager API
   - Container Registry API

## Architecture Overview

- **Compute**: Cloud Run (serverless containers)
- **Database**: Cloud SQL (PostgreSQL)
- **Storage**: Cloud Storage (images and thumbnails)
- **Secrets**: Secret Manager (passwords and sensitive data)

## Step-by-Step Deployment

### 1. Enable Required APIs

```bash
gcloud services enable run.googleapis.com \
  cloudbuild.googleapis.com \
  sqladmin.googleapis.com \
  secretmanager.googleapis.com \
  containerregistry.googleapis.com \
  storage.googleapis.com
```

### 2. Set Environment Variables

```bash
export PROJECT_ID=$(gcloud config get-value project)
export REGION=us-central1
export SERVICE_NAME=photowebsite
export DB_INSTANCE_NAME=photowebsite-db
export BUCKET_NAME=${PROJECT_ID}-photowebsite
```

### 3. Create Cloud SQL Instance

```bash
# Create PostgreSQL instance (this takes 5-10 minutes)
gcloud sql instances create $DB_INSTANCE_NAME \
  --database-version=POSTGRES_15 \
  --tier=db-f1-micro \
  --region=$REGION \
  --storage-type=SSD \
  --storage-size=10GB \
  --storage-auto-increase

# Set the postgres user password
gcloud sql users set-password postgres \
  --instance=$DB_INSTANCE_NAME \
  --password="YOUR_SECURE_DB_PASSWORD"

# Create the database
gcloud sql databases create photowebsite \
  --instance=$DB_INSTANCE_NAME
```

### 4. Create Cloud Storage Bucket

```bash
# Create bucket for image storage
gsutil mb -l $REGION gs://$BUCKET_NAME

# Make bucket publicly readable (for serving images)
gsutil iam ch allUsers:objectViewer gs://$BUCKET_NAME

# Enable uniform bucket-level access
gsutil uniformbucketlevelaccess set on gs://$BUCKET_NAME

# Set CORS configuration to allow access from your domain
cat > cors.json << EOF
[
  {
    "origin": ["*"],
    "method": ["GET", "HEAD"],
    "responseHeader": ["Content-Type"],
    "maxAgeSeconds": 3600
  }
]
EOF
gsutil cors set cors.json gs://$BUCKET_NAME
rm cors.json
```

### 5. Store Secrets in Secret Manager

```bash
# Create admin password secret
echo -n "YOUR_ADMIN_PASSWORD" | gcloud secrets create photowebsite-admin-password \
  --data-file=- \
  --replication-policy="automatic"

# Create database password secret
echo -n "YOUR_SECURE_DB_PASSWORD" | gcloud secrets create photowebsite-db-password \
  --data-file=- \
  --replication-policy="automatic"

# Grant Cloud Run access to secrets
PROJECT_NUMBER=$(gcloud projects describe $PROJECT_ID --format="value(projectNumber)")
gcloud secrets add-iam-policy-binding photowebsite-admin-password \
  --member="serviceAccount:${PROJECT_NUMBER}-compute@developer.gserviceaccount.com" \
  --role="roles/secretmanager.secretAccessor"

gcloud secrets add-iam-policy-binding photowebsite-db-password \
  --member="serviceAccount:${PROJECT_NUMBER}-compute@developer.gserviceaccount.com" \
  --role="roles/secretmanager.secretAccessor"
```

### 6. Build and Deploy with Cloud Build

Update the `cloudbuild.yaml` substitutions:

```yaml
substitutions:
  _DEPLOY_REGION: 'us-central1'
  _GCS_BUCKET_NAME: 'your-project-id-photowebsite'
  _CLOUDSQL_INSTANCE: 'your-project-id:us-central1:photowebsite-db'
```

Then deploy:

```bash
gcloud builds submit --config=cloudbuild.yaml \
  --substitutions=_DEPLOY_REGION=$REGION,_GCS_BUCKET_NAME=$BUCKET_NAME,_CLOUDSQL_INSTANCE=${PROJECT_ID}:${REGION}:${DB_INSTANCE_NAME}
```

### 7. Alternative: Manual Docker Build and Deploy

If you prefer manual deployment:

```bash
# Build the Docker image
docker build -t gcr.io/$PROJECT_ID/photowebsite .

# Push to Container Registry
docker push gcr.io/$PROJECT_ID/photowebsite

# Deploy to Cloud Run
gcloud run deploy $SERVICE_NAME \
  --image gcr.io/$PROJECT_ID/photowebsite \
  --region $REGION \
  --platform managed \
  --allow-unauthenticated \
  --set-env-vars="USE_POSTGRES=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=$BUCKET_NAME,DB_NAME=photowebsite,DB_USER=postgres,INSTANCE_UNIX_SOCKET=/cloudsql/${PROJECT_ID}:${REGION}:${DB_INSTANCE_NAME}" \
  --set-secrets="ADMIN_PASSWORD=photowebsite-admin-password:latest,DB_PASSWORD=photowebsite-db-password:latest" \
  --add-cloudsql-instances=${PROJECT_ID}:${REGION}:${DB_INSTANCE_NAME} \
  --memory=512Mi \
  --cpu=1 \
  --max-instances=10 \
  --min-instances=0
```

### 8. Get Your Service URL

```bash
gcloud run services describe $SERVICE_NAME --region=$REGION --format='value(status.url)'
```

Your application should now be live!

## Post-Deployment Configuration

### Custom Domain (Optional)

```bash
# Map a custom domain
gcloud run domain-mappings create --service=$SERVICE_NAME --domain=yourdomain.com --region=$REGION
```

### Environment Variables

To update environment variables after deployment:

```bash
gcloud run services update $SERVICE_NAME \
  --region=$REGION \
  --set-env-vars="NEW_VAR=value"
```

### Scaling Configuration

```bash
# Update scaling settings
gcloud run services update $SERVICE_NAME \
  --region=$REGION \
  --min-instances=1 \
  --max-instances=20 \
  --cpu=2 \
  --memory=1Gi
```

## Cost Optimization

### Development/Testing Environment

For a low-cost development environment:
- Use `db-f1-micro` tier (smallest, ~$10/month)
- Set `--min-instances=0` (pay only for usage)
- Use `--memory=256Mi` if sufficient

### Production Environment

For better performance:
- Use `db-g1-small` or higher tier
- Set `--min-instances=1` to reduce cold starts
- Use `--memory=512Mi` or higher
- Consider Cloud CDN for static assets

### Free Tier

Cloud Run offers:
- 2 million requests per month (free)
- 360,000 GB-seconds memory (free)
- 180,000 vCPU-seconds (free)

## Monitoring and Logs

```bash
# View logs
gcloud run services logs read $SERVICE_NAME --region=$REGION

# Monitor in Cloud Console
https://console.cloud.google.com/run/detail/$REGION/$SERVICE_NAME
```

## Backup and Maintenance

### Database Backups

Cloud SQL automatically backs up your database. To create a manual backup:

```bash
gcloud sql backups create --instance=$DB_INSTANCE_NAME
```

### Storage Backups

```bash
# List objects
gsutil ls -r gs://$BUCKET_NAME

# Sync to another bucket for backup
gsutil -m rsync -r gs://$BUCKET_NAME gs://${BUCKET_NAME}-backup
```

## Local Development

For local development, keep using SQLite and local filesystem:

```bash
# .env file for local development
USE_POSTGRES=false
USE_CLOUD_STORAGE=false
ADMIN_PASSWORD=changeme
```

Then run:
```bash
npm install
npm run build:css
npm start
```

## Troubleshooting

### Database Connection Issues

1. Verify Cloud SQL instance is running:
   ```bash
   gcloud sql instances describe $DB_INSTANCE_NAME
   ```

2. Check Cloud Run service has Cloud SQL connection:
   ```bash
   gcloud run services describe $SERVICE_NAME --region=$REGION
   ```

3. Verify secrets are accessible:
   ```bash
   gcloud secrets versions access latest --secret=photowebsite-db-password
   ```

### Storage Issues

1. Verify bucket exists and is public:
   ```bash
   gsutil ls gs://$BUCKET_NAME
   gsutil iam get gs://$BUCKET_NAME
   ```

2. Check service account permissions:
   ```bash
   gsutil iam get gs://$BUCKET_NAME
   ```

### Application Errors

Check the logs:
```bash
gcloud run services logs tail $SERVICE_NAME --region=$REGION
```

## CI/CD with Cloud Build Triggers

Set up automatic deployment on git push:

```bash
gcloud builds triggers create github \
  --repo-name=PhotoWebsite \
  --repo-owner=YOUR_GITHUB_USERNAME \
  --branch-pattern="^main$" \
  --build-config=cloudbuild.yaml
```

## Security Best Practices

1. **Never commit secrets** to version control
2. **Use Secret Manager** for all sensitive data
3. **Enable IAM authentication** for admin routes in production
4. **Use HTTPS only** (Cloud Run enforces this by default)
5. **Set up Cloud Armor** for DDoS protection if needed
6. **Regular updates** of dependencies

## Cleanup

To delete all resources:

```bash
# Delete Cloud Run service
gcloud run services delete $SERVICE_NAME --region=$REGION

# Delete Cloud SQL instance
gcloud sql instances delete $DB_INSTANCE_NAME

# Delete Cloud Storage bucket (removes all files!)
gsutil -m rm -r gs://$BUCKET_NAME

# Delete secrets
gcloud secrets delete photowebsite-admin-password
gcloud secrets delete photowebsite-db-password
```

## Support

For issues related to:
- **Google Cloud Platform**: https://cloud.google.com/support
- **Application bugs**: Check the logs and GitHub issues

## Estimated Costs

### Minimal Usage (Development)
- Cloud Run: ~$0-5/month (within free tier)
- Cloud SQL (f1-micro): ~$10/month
- Cloud Storage: ~$1/month (for 10GB)
- **Total: ~$11-16/month**

### Medium Usage (Production)
- Cloud Run: ~$10-30/month
- Cloud SQL (g1-small): ~$25/month
- Cloud Storage: ~$3-10/month
- **Total: ~$38-65/month**

Actual costs depend on traffic, storage, and compute requirements.

