<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Read SP Page Builder Page Action
 *
 * @since  1.0.0
 */
class ReadPageBuilderPageAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Page data
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for ReadPageBuilderPageAction');
        }

        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }
        $pageId = $parameters['page_id'];

        // Check if SP Page Builder is installed
        if (!file_exists(JPATH_ROOT . '/components/com_sppagebuilder')) {
            throw new \RuntimeException('SP Page Builder is not installed');
        }

        // Fetch the page from #__sppagebuilder (not _pages)
        $query = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__sppagebuilder'))
            ->where($db->quoteName('id') . ' = ' . (int) $pageId);

        $db->setQuery($query);
        $page = $db->loadObject();

        if (!$page) {
            throw new \RuntimeException("SP Page Builder page with ID {$pageId} not found");
        }

        // Fetch addons/sections for this page
        $addonQuery = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__sppagebuilder_addons'))
            ->where($db->quoteName('page_id') . ' = ' . (int) $pageId)
            ->order($db->quoteName('ordering') . ' ASC');
        
        $db->setQuery($addonQuery);
        $addons = $db->loadObjectList();

        // Decode the page text if JSON
        $pageContent = null;
        if (!empty($page->text)) {
            $decoded = json_decode($page->text, true);
            $pageContent = $decoded ? $decoded : $page->text;
        }

        return [
            'id' => $page->id,
            'title' => $page->title,
            'text' => $page->text,
            'content' => $pageContent,
            'addons' => $addons,
            'addon_count' => count($addons),
            'published' => $page->published,
            'created_on' => $page->created_on ?? null,
            'modified' => $page->modified ?? null,
            'language' => $page->language ?? '*',
            'access' => $page->access ?? 1
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Read a SP Page Builder page by ID. Returns page data including title, content sections, and metadata.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['page_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['page_id']) && is_numeric($parameters['page_id']);
    }
}

