<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Read SP Page Builder Page Action
 *
 * @since  1.0.0
 */
class ReadPageBuilderPageAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Page data
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for ReadPageBuilderPageAction');
        }

        $db = Factory::getContainer()->get('DatabaseDriver');
        $pageId = $parameters['page_id'];

        // Check if SP Page Builder is installed
        if (!file_exists(JPATH_ROOT . '/components/com_sppagebuilder')) {
            throw new \RuntimeException('SP Page Builder is not installed');
        }

        $query = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__sppagebuilder_pages'))
            ->where($db->quoteName('id') . ' = ' . (int) $pageId);

        $db->setQuery($query);
        $page = $db->loadObject();

        if (!$page) {
            throw new \RuntimeException("SP Page Builder page with ID {$pageId} not found");
        }

        // Decode the page content
        $content = json_decode($page->text, true);

        return [
            'id' => $page->id,
            'title' => $page->title,
            'content' => $content,
            'published' => $page->published,
            'created_on' => $page->created_on,
            'modified' => $page->modified,
            'language' => $page->language,
            'access' => $page->access
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Read a SP Page Builder page by ID. Returns page data including title, content sections, and metadata.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['page_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['page_id']) && is_numeric($parameters['page_id']);
    }
}

