(function(){
  function ready(fn){ if(document.readyState!=='loading') fn(); else document.addEventListener('DOMContentLoaded', fn); }

  ready(function(){
    // Handle each gallery section separately
    const gallerySections = document.querySelectorAll('.gallery');
    
    gallerySections.forEach(function(gallery) {
      const galleryLinks = Array.from(gallery.querySelectorAll('a'));
      if (galleryLinks.length === 0) return;

      const imageUrls = galleryLinks.map(a => a.getAttribute('href'));
      let currentIndex = 0;
      
      // Check if URL has a photo hash to auto-open
      const urlHash = window.location.hash;
      const photoMatch = urlHash.match(/^#photo-(\d+)$/);
      let autoOpenIndex = null;
      if (photoMatch) {
        const photoNum = parseInt(photoMatch[1], 10);
        if (photoNum >= 1 && photoNum <= imageUrls.length) {
          autoOpenIndex = photoNum - 1;
        }
      }

      // Build overlay for this gallery
      const overlay = document.createElement('div');
      overlay.className = 'lightbox';
      overlay.innerHTML = `
        <button class="lightbox__close" aria-label="Close">×</button>
        <button class="lightbox__prev" aria-label="Previous">‹</button>
        <div class="lightbox__img-container">
          <img class="lightbox__img" alt="" />
          <div class="lightbox__actions">
            <button class="lightbox__action" id="lightbox-download" aria-label="Download" title="Download">
              <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                <polyline points="7 10 12 15 17 10"></polyline>
                <line x1="12" y1="15" x2="12" y2="3"></line>
              </svg>
            </button>
            <button class="lightbox__action" id="lightbox-share" aria-label="Share" title="Share">
              <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                <circle cx="18" cy="5" r="3"></circle>
                <circle cx="6" cy="12" r="3"></circle>
                <circle cx="18" cy="19" r="3"></circle>
                <line x1="8.59" y1="13.51" x2="15.42" y2="17.49"></line>
                <line x1="15.41" y1="6.51" x2="8.59" y2="10.49"></line>
              </svg>
            </button>
          </div>
        </div>
        <button class="lightbox__next" aria-label="Next">›</button>
      `;
      document.body.appendChild(overlay);

      const img = overlay.querySelector('.lightbox__img');
      const btnClose = overlay.querySelector('.lightbox__close');
      const btnPrev = overlay.querySelector('.lightbox__prev');
      const btnNext = overlay.querySelector('.lightbox__next');
      const btnDownload = overlay.querySelector('#lightbox-download');
      const btnShare = overlay.querySelector('#lightbox-share');

      function updateNavVisibility(){
        if (imageUrls.length <= 1) {
          btnPrev.style.display = 'none';
          btnNext.style.display = 'none';
        } else {
          btnPrev.style.display = '';
          btnNext.style.display = '';
        }
      }

      function openAt(index, updateHash = false){
        currentIndex = Math.max(0, Math.min(index, imageUrls.length - 1));
        img.src = imageUrls[currentIndex];
        overlay.classList.add('lightbox--open');
        updateNavVisibility();
        document.addEventListener('keydown', onKey);
        
        // Update URL hash when navigating
        if (updateHash) {
          window.history.replaceState(null, '', `#photo-${currentIndex + 1}`);
        }
      }

      function close(){
        overlay.classList.remove('lightbox--open');
        document.removeEventListener('keydown', onKey);
        // Remove hash when closing
        if (window.location.hash.startsWith('#photo-')) {
          window.history.replaceState(null, '', window.location.pathname);
        }
      }

      function prev(){ openAt((currentIndex - 1 + imageUrls.length) % imageUrls.length, true); }
      function next(){ openAt((currentIndex + 1) % imageUrls.length, true); }

      function onKey(e){
        if (e.key === 'Escape') close();
        else if (e.key === 'ArrowLeft') prev();
        else if (e.key === 'ArrowRight') next();
      }

      function downloadImage(){
        const url = imageUrls[currentIndex];
        const filename = url.split('/').pop();
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
      }

      function shareImage(){
        // Create direct link to this specific photo in the album
        const baseUrl = window.location.origin + window.location.pathname;
        const photoNumber = currentIndex + 1;
        const directUrl = `${baseUrl}#photo-${photoNumber}`;
        const albumTitle = document.title.replace(' • Lucky Pup', '');
        const totalPhotos = imageUrls.length;
        
        if (navigator.share) {
          // Use native share menu (works on mobile)
          navigator.share({
            title: `${albumTitle} - Photo ${photoNumber}`,
            text: `Check out photo ${photoNumber} of ${totalPhotos} from ${albumTitle}`,
            url: directUrl
          }).catch(err => {
            if (err.name !== 'AbortError') {
              console.log('Share error:', err);
            }
          });
        } else {
          // Fallback: copy direct photo URL to clipboard
          navigator.clipboard.writeText(directUrl).then(() => {
            const originalText = btnShare.innerHTML;
            btnShare.innerHTML = '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M20 6L9 17l-5-5"></path></svg>';
            setTimeout(() => {
              btnShare.innerHTML = originalText;
            }, 2000);
          }).catch(() => {
            alert('Link: ' + directUrl);
          });
        }
      }

      overlay.addEventListener('click', function(e){
        // close when clicking outside the image and buttons
        const target = e.target;
        if (target === overlay) close();
      });
      btnClose.addEventListener('click', close);
      btnPrev.addEventListener('click', prev);
      btnNext.addEventListener('click', next);
      btnDownload.addEventListener('click', downloadImage);
      btnShare.addEventListener('click', shareImage);

      galleryLinks.forEach(function(a, idx){
        a.addEventListener('click', function(ev){
          ev.preventDefault();
          openAt(idx, true);
        });
      });
      
      // Auto-open photo if hash is present in URL
      if (autoOpenIndex !== null) {
        setTimeout(() => openAt(autoOpenIndex), 100);
      }
    });
  });
})();