<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Seasons;
use Livewire\Component;
use Illuminate\Support\Facades\Cache;

class DashboardFilters extends Component
{
    public $selectedSeason = null;
    public $selectedOrderType = 'all';
    public $seasons = [];
    public $orderTypes = [
        'all' => 'All Orders',
        'wholesale' => 'Wholesale',
        'commission' => 'Commission'
    ];

    public function mount()
    {
        $this->seasons = Seasons::orderBy('created_at', 'desc')->get();
        
        // Get saved preferences from user settings
        $savedSeason = auth()->user()->setting('dashboard_season_filter');
        $savedOrderType = auth()->user()->setting('dashboard_order_type_filter');
        
        if ($savedSeason && $this->seasons->contains('id', $savedSeason)) {
            $this->selectedSeason = $savedSeason;
        } else {
            // Default to 4 seasons back from the latest
            $this->selectedSeason = $this->seasons->skip(3)->first()?->id ?? $this->seasons->first()?->id;
        }
        
        if ($savedOrderType && array_key_exists($savedOrderType, $this->orderTypes)) {
            $this->selectedOrderType = $savedOrderType;
        }
    }

    public function updatedSelectedSeason()
    {
        // Save the season preference to user settings
        auth()->user()->settings(['dashboard_season_filter' => $this->selectedSeason]);
        
        // Clear cache and refresh all dashboard components
        $this->clearDashboardCache();
        $this->dispatch('dashboardFiltersUpdated', [
            'season' => $this->selectedSeason,
            'orderType' => $this->selectedOrderType
        ]);
    }

    public function updatedSelectedOrderType()
    {
        // Save the order type preference to user settings
        auth()->user()->settings(['dashboard_order_type_filter' => $this->selectedOrderType]);
        
        // Clear cache and refresh all dashboard components
        $this->clearDashboardCache();
        $this->dispatch('dashboardFiltersUpdated', [
            'season' => $this->selectedSeason,
            'orderType' => $this->selectedOrderType
        ]);
    }

    public function refreshDashboard()
    {
        // Clear all dashboard cache
        $this->clearDashboardCache();
        
        // Dispatch refresh event to all components
        $this->dispatch('refreshDashboard');
    }

    private function clearDashboardCache()
    {
        // Clear all dashboard-related cache keys
        $cacheKeys = [
            'dashboard_total_styles',
            'dashboard_total_qmts_shipped',
            'dashboard_total_gmts_left_to_ship',
            'dashboard_total_value_shipped',
            'dashboard_total_value_ordered',
            'dashboard_total_samples_sent',
            'dashboard_total_invoices_sent',
            'garments_due_chart_data'
        ];
        
        foreach ($cacheKeys as $key) {
            Cache::forget($key);
        }
        
        // Clear customer sales cache for all seasons and order types
        $seasons = Seasons::pluck('id');
        foreach ($seasons as $seasonId) {
            foreach (['all', 'wholesale', 'commission'] as $orderType) {
                Cache::forget("customer_sales_dashboard_v3_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_qmts_shipped_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_gmts_left_to_ship_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_value_shipped_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_value_ordered_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_samples_sent_season_{$seasonId}_order_type_{$orderType}");
                Cache::forget("dashboard_total_invoices_sent_season_{$seasonId}_order_type_{$orderType}");
            }
        }
    }

    public function render()
    {
        return view('livewire.dashboard.dashboard-filters');
    }
}
