# Routing Architecture

## Overview

The system now uses a three-tier subdomain architecture:

```
www.vybephoto.com (or vybephoto.com)
  ↓ Main website (Joomla) - Not handled by Node.js

app.vybephoto.com
  ↓ Photographer portal (Node.js)
  ↓ Login, Register, Dashboard, Admin

john-smith.vybephoto.com
  ↓ Client gallery (Node.js)
  ↓ View albums, photos, groups
```

## Subdomain Routing

### 1. **Main Domain** (`www.vybephoto.com` or `vybephoto.com`)

**Served by:** Joomla (external CMS)

**Purpose:** Marketing website, company information

**Node.js behavior:** Shows a holding page explaining that the main site should be served by Joomla, with a link to the app portal.

**Routes:**
- All routes handled by Joomla (configure reverse proxy)

---

### 2. **App Subdomain** (`app.vybephoto.com`)

**Served by:** Node.js application

**Purpose:** Photographer authentication, dashboard, and admin panel

**Routes:**

#### Public Routes (Not Logged In)
- `GET /` → Landing page with login/register (shows pricing tiers)
- `GET /register` → Registration form
- `POST /register` → Handle registration
- `GET /login` → Login form
- `POST /login` → Handle login

#### Protected Routes (Logged In)
- `GET /dashboard` → Photographer dashboard (usage stats, quick actions)
- `GET /admin` → Album management
- `GET /admin/settings` → Branding settings
- `GET /admin/albums/:id` → Edit album
- `POST /admin/albums/:id/upload` → Upload photos
- All other admin routes...

**Behavior:**
- ✅ If logged in, `/` redirects to `/dashboard`
- ✅ If not logged in, shows marketing landing page
- ✅ All auth routes ONLY work on this subdomain
- ✅ Trying to access `/register` or `/login` on other subdomains returns 404

---

### 3. **User Subdomains** (`username.vybephoto.com`)

**Served by:** Node.js application

**Purpose:** Individual photographer's client-facing gallery

**Routes:**

#### Public Routes
- `GET /` → Gallery homepage (shows public albums or homepage album)
- `POST /access` → Submit access code for private album/group
- `GET /a/:code` → View album by code
- `GET /g/:code` → View group by code
- `POST /verify-age` → Age verification (if enabled)
- `GET /thumb/:albumId/:filename` → Thumbnail images

**Behavior:**
- ✅ Loads photographer's branding (logo, colors, site name)
- ✅ Shows only this photographer's albums/photos
- ✅ Age verification if photographer enabled it
- ✅ No access to other photographers' data
- ✅ No admin routes available here

---

### 4. **Custom Domains** (e.g., `johnsmithphotography.com`)

**Served by:** Node.js application

**Purpose:** Same as user subdomains, but with photographer's own domain

**Routes:** Same as user subdomains

**Behavior:**
- ✅ Professional/Business tier feature
- ✅ DNS CNAME points to their subdomain
- ✅ Domain must be verified before activation
- ✅ Functions identically to username.vybephoto.com

---

## Environment Variables

```bash
# Required
BASE_DOMAIN=vybephoto.com
APP_SUBDOMAIN=app

# Optional (legacy)
ADMIN_SUBDOMAIN=app  # Same as APP_SUBDOMAIN
```

---

## Local Testing Setup

### 1. Edit `/etc/hosts` (macOS/Linux)

```bash
sudo nano /etc/hosts
```

Add these lines:
```
127.0.0.1 localhost
127.0.0.1 www.vybephoto.com
127.0.0.1 app.localhost
127.0.0.1 app.vybephoto.com
127.0.0.1 john-smith.localhost
127.0.0.1 john-smith.vybephoto.com
127.0.0.1 test-photographer.localhost
127.0.0.1 test-photographer.vybephoto.com
```

### 2. Start the Server

```bash
npm run start:multi
```

### 3. Test Each Subdomain

**Main Domain:**
```bash
http://www.vybephoto.com:3000
```
Should show: "This is the Node.js backend, main site should be Joomla"

**App Portal (Registration/Login):**
```bash
http://app.vybephoto.com:3000
http://app.localhost:3000
```
Should show: Beautiful landing page with "Start Free Trial" and pricing tiers

**User Gallery (after registration):**
```bash
http://john-smith.vybephoto.com:3000
http://john-smith.localhost:3000
```
Should show: John's branded gallery with his albums

---

## Middleware Flow

```
Request arrives
    ↓
detectTenant middleware
    ↓ Checks hostname
    ├─ www or base domain → req.tenant = null, isMainSite = true
    ├─ app.domain → req.tenant = null, isAppSubdomain = true
    └─ username.domain → Loads user from DB → req.tenant = user object
    ↓
authenticateUser middleware
    ↓ Checks session cookie
    └─ Sets req.user if valid session
    ↓
Route handlers
    ├─ requireAppSubdomain → Only allow on app subdomain
    ├─ requireAuth → Must be logged in
    └─ requireTenant → Must have valid photographer loaded
```

---

## Security

### Data Isolation

✅ **Tenant Detection:** Every request identifies which photographer's site it is
✅ **Database Queries:** All queries filtered by `user_id`
✅ **File Storage:** Files stored in `userId/albumId/filename` paths
✅ **Sessions:** User can only access their own data

### Authentication

✅ **Login/Register:** Only available on `app.subdomain`
✅ **Dashboard/Admin:** Only available on `app.subdomain` + requires login
✅ **Gallery Access:** Public or code-protected albums on user subdomains
✅ **Session Cookies:** HTTP-only, 30-day expiration, secure in production

---

## Production Deployment

### DNS Configuration

**For main domain (Joomla):**
```
Type: A
Name: @ (or www)
Value: <Joomla server IP>
```

**For Node.js app (wildcard):**
```
Type: A
Name: *
Value: <Node.js server IP>
```

**Or use separate records:**
```
Type: A
Name: app
Value: <Node.js server IP>

Type: A
Name: *
Value: <Node.js server IP>
```

### Reverse Proxy (nginx example)

```nginx
# Main site (Joomla)
server {
    listen 80;
    server_name vybephoto.com www.vybephoto.com;
    
    location / {
        proxy_pass http://joomla_server:80;
    }
}

# App portal and user galleries (Node.js)
server {
    listen 80;
    server_name app.vybephoto.com *.vybephoto.com;
    
    location / {
        proxy_pass http://nodejs_server:3000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    }
}
```

### SSL/HTTPS

Use Let's Encrypt with wildcard certificate:

```bash
certbot certonly --dns-cloudflare \
  -d vybephoto.com \
  -d *.vybephoto.com
```

---

## User Journey Examples

### 1. Photographer Signs Up

```
1. Visits: app.vybephoto.com
2. Clicks "Start Free Trial"
3. Fills form with username "john-smith"
4. Submits → Account created
5. Redirected to: app.vybephoto.com/dashboard
6. Sees: Usage stats, quick actions
7. Clicks "My Albums" → app.vybephoto.com/admin
8. Creates album → Gets access code
9. Views gallery at: john-smith.vybephoto.com
```

### 2. Client Views Photos

```
1. Receives link: john-smith.vybephoto.com/a/ABC12345
2. Visits link → Sees John's branded gallery
3. Enters access code (if private)
4. Views/downloads photos
5. Never needs to create account
```

### 3. Photographer Manages Gallery

```
1. Visits: app.vybephoto.com
2. Logs in
3. Dashboard shows: 234/1,000 photos used
4. Clicks "My Albums"
5. Uploads photos
6. Shares link: john-smith.vybephoto.com/a/XYZ789
```

---

## Troubleshooting

### "This page is only available at app.vybephoto.com"

**Problem:** Trying to access `/register`, `/login`, or `/dashboard` from wrong subdomain

**Solution:** Navigate to `app.vybephoto.com` instead

### "Gallery Not Found"

**Problem:** Username doesn't exist in database

**Solution:** 
1. Check username is spelled correctly
2. Ensure user has registered at `app.vybephoto.com/register`
3. Verify user exists in `users` table

### Can't access localhost

**Problem:** Browser doesn't resolve subdomains locally

**Solution:** Add entries to `/etc/hosts` as shown above

### Shows Joomla message on app subdomain

**Problem:** DNS or reverse proxy misconfigured

**Solution:** Ensure `app.domain` points to Node.js server, not Joomla

---

## Summary

| Domain | Purpose | Served By | Authentication | Admin Access |
|--------|---------|-----------|----------------|--------------|
| `www.vybephoto.com` | Marketing | Joomla | N/A | N/A |
| `app.vybephoto.com` | Portal | Node.js | ✅ Login/Register | ✅ Full Admin |
| `user.vybephoto.com` | Gallery | Node.js | ❌ Public/Code | ❌ View Only |
| `custom.com` | Gallery | Node.js | ❌ Public/Code | ❌ View Only |

**Key Principle:** Authentication and admin features are isolated to `app.subdomain`, while galleries are public-facing at `username.subdomain`.

