<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Rule;
use Smalot\PdfParser\Parser;

class ProcessInxpressInvoice extends Component
{
    use WithFileUploads;

    #[Rule('required|file|mimes:pdf|max:10240')] // 10MB max
    public $pdfFile;
    public $shipments = [];
    public $grandTotal = 0.0;
    public $fileName = null;
    public $fileSizeKB = null;
    public $grandVat = 0.0;
    public $forceTabula = true; // Enforce table extraction path during debugging

    public function processInvoice()
    {
        $this->validate();

        if ($this->pdfFile) {
            try {
                // Store the file temporarily
                $path = $this->pdfFile->store('temp');
                
                // Get the full path - store() returns relative path from storage/app/
                $fullPath = storage_path('app/' . $path);
                
                // Debug: Log the paths to see what's happening
                \Log::info("Stored path: " . $path);
                \Log::info("Full path: " . $fullPath);
                \Log::info("File exists: " . (file_exists($fullPath) ? 'YES' : 'NO'));
                
                // Alternative: try using the temporary path directly
                if (!file_exists($fullPath)) {
                    $altPath = $this->pdfFile->getRealPath();
                    \Log::info("Alternative path: " . $altPath);
                    if ($altPath && file_exists($altPath)) {
                        $fullPath = $altPath;
                        \Log::info("Using alternative path: " . $fullPath);
                    }
                }
                
                // Cache file metadata for UI before any temp moves
                try {
                    $this->fileName = $this->pdfFile->getClientOriginalName();
                } catch (\Throwable $e) {
                    $this->fileName = null;
                }
                try {
                    $this->fileSizeKB = round(((float) $this->pdfFile->getSize()) / 1024, 2);
                } catch (\Throwable $e) {
                    $this->fileSizeKB = null;
                }

                // Prefer table extraction via Tabula; ONLY fallback to text if Tabula fails entirely
                $shipments = [];
                $usedTabula = false;
                try {
                    $tables = $this->extractTablesWithTabula($fullPath);
                    \Log::info('Tabula extracted tables: ' . (is_array($tables) ? count($tables) : 0));
                    if (!empty($tables)) {
                        $usedTabula = true;
                        $shipments = $this->buildShipmentsFromTables($tables);
                        \Log::info('Tabula-built shipments: ' . count($shipments));
                    }
                } catch (\Throwable $e) {
                    \Log::warning('Tabula extraction failed: ' . $e->getMessage());
                }

                if (!$usedTabula) {
                    if ($this->forceTabula) {
                        throw new \RuntimeException('Tabula did not return tables. Aborting due to forceTabula. Check Java/shell_exec/JAR.');
                    }
                    // Extract text from PDF (fallback)
                    $text = $this->extractTextFromPdf($fullPath);
                    // Parse the PDF text and extract shipment data
                    $shipments = $this->parseInvoiceData($text);
                    \Log::info('Used text parser fallback, shipments: ' . count($shipments));
                }

                if ($this->forceTabula && empty($shipments)) {
                    throw new \RuntimeException('Tabula returned tables but produced zero shipments. Aborting due to forceTabula.');
                }

                // Clean up the temporary file
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                }
                
                // Expose to UI
                $this->shipments = $shipments;
                $this->grandTotal = 0.0;
                $this->grandVat = 0.0;
                foreach ($shipments as $s) {
                    $this->grandTotal += isset($s['total']) ? (float) $s['total'] : 0.0;
                    $this->grandVat += isset($s['vat_amount']) ? (float) $s['vat_amount'] : 0.0;
                }

            } catch (\Exception $e) {
                // Clean up the temporary file if it exists
                if (isset($fullPath) && file_exists($fullPath)) {
                    unlink($fullPath);
                }
                
                // Show error
                dd("Error processing PDF: " . $e->getMessage());
            }
        }
    }

    /**
     * Ensure Tabula-Java JAR exists locally; download if missing.
     */
    private function ensureTabulaJarPath(): string
    {
        $dir = storage_path('app/tabula');
        if (!is_dir($dir)) {
            @mkdir($dir, 0775, true);
        }
        $jarPath = $dir . DIRECTORY_SEPARATOR . 'tabula-1.0.5-jar-with-dependencies.jar';
        if (!file_exists($jarPath)) {
            $url = 'https://github.com/tabulapdf/tabula-java/releases/download/v1.0.5/tabula-1.0.5-jar-with-dependencies.jar';
            $tmp = $jarPath . '.part';
            $ctx = stream_context_create([
                'http' => ['timeout' => 120],
                'https' => ['timeout' => 120],
            ]);
            $data = @file_get_contents($url, false, $ctx);
            if ($data === false) {
                throw new \RuntimeException('Failed to download Tabula JAR. Ensure internet access or pre-provision the JAR at ' . $jarPath);
            }
            file_put_contents($tmp, $data);
            rename($tmp, $jarPath);
        }
        return $jarPath;
    }

    private function hasJavaInstalled(): bool
    {
        $which = function_exists('shell_exec') ? (shell_exec('command -v java 2>&1') ?? '') : '';
        return trim($which) !== '';
    }

    private function getJavaBinary(): string
    {
        // Prefer Homebrew path if present to avoid PATH issues in php-fpm
        $brewJava = '/opt/homebrew/opt/openjdk@17/bin/java';
        if (is_executable($brewJava)) {
            return $brewJava;
        }
        // Fallback to PATH-resolved java
        return 'java';
    }

    /**
     * Run Tabula-Java to extract tables from the PDF across all pages.
     * Returns an array of tables, each table as array<array<string>> rows.
     */
    private function extractTablesWithTabula(string $pdfPath): array
    {
        if (!function_exists('shell_exec')) {
            throw new \RuntimeException('shell_exec is disabled; Tabula requires shell access.');
        }
        if (!$this->hasJavaInstalled()) {
            throw new \RuntimeException('Java is not installed or not on PATH.');
        }
        $jar = $this->ensureTabulaJarPath();
        $workDir = storage_path('app/tabula');
        if (!is_dir($workDir)) {
            @mkdir($workDir, 0775, true);
        }
        $uniq = uniqid('tabula_', true);
        $outFile = $workDir . DIRECTORY_SEPARATOR . ('out_' . $uniq . '.json');

        $escapedJar = escapeshellarg($jar);
        $escapedPdf = escapeshellarg($pdfPath);
        $escapedOut = escapeshellarg($outFile);

        $javaBin = escapeshellcmd($this->getJavaBinary());
        $cmdLattice = $javaBin . ' -Dfile.encoding=UTF8 -jar ' . $escapedJar . ' -p all --lattice --guess -f JSON -o ' . $escapedOut . ' ' . $escapedPdf . ' 2>&1';
        $outputLattice = function_exists('shell_exec') ? (shell_exec($cmdLattice) ?? '') : '';
        $tables = $this->parseTabulaJsonToTables($outFile);
        if (empty($tables)) {
            @unlink($outFile);
            $cmdStream = $javaBin . ' -Dfile.encoding=UTF8 -jar ' . $escapedJar . ' -p all --stream --guess -f JSON -o ' . $escapedOut . ' ' . $escapedPdf . ' 2>&1';
            $outputStream = function_exists('shell_exec') ? (shell_exec($cmdStream) ?? '') : '';
            $tables = $this->parseTabulaJsonToTables($outFile);
        }

        if (!empty($tables)) {
            // Success: remove temp JSON
            if (file_exists($outFile)) {
                @unlink($outFile);
            }
            $this->logTabulaPreview($tables);
        } else {
            // Failure: keep JSON and write a debug log
            $debugPath = storage_path('logs/tabula_debug_' . $uniq . '.log');
            $debug = [];
            $debug[] = 'PDF: ' . $pdfPath;
            $debug[] = 'Jar: ' . $jar;
            $debug[] = 'Out JSON: ' . (file_exists($outFile) ? $outFile : '(not created)');
            $debug[] = 'Java bin: ' . $this->getJavaBinary();
            $debug[] = 'Lattice CMD: ' . $cmdLattice;
            $debug[] = 'Lattice OUT (first 500): ' . substr($outputLattice, 0, 500);
            if (isset($outputStream)) {
                $debug[] = 'Stream CMD: ' . $cmdStream;
                $debug[] = 'Stream OUT (first 500): ' . substr($outputStream, 0, 500);
            }
            @file_put_contents($debugPath, implode("\n\n", $debug));
            \Log::warning('Tabula produced no tables. Debug log at ' . $debugPath . ' JSON at ' . $outFile);
        }

        return $tables;
    }

    /**
     * Parse Tabula JSON output file into an array of tables (rows of cell strings).
     */
    private function parseTabulaJsonToTables(string $jsonPath): array
    {
        if (!file_exists($jsonPath)) {
            return [];
        }
        $json = @file_get_contents($jsonPath);
        if ($json === false || $json === '') {
            return [];
        }
        $decoded = json_decode($json, true);
        if (!is_array($decoded)) {
            return [];
        }

        $rawTables = [];
        if (array_is_list($decoded)) {
            $rawTables = $decoded;
        } elseif (isset($decoded['tables']) && is_array($decoded['tables'])) {
            $rawTables = $decoded['tables'];
        } else {
            $rawTables = [$decoded];
        }

        $tables = [];
        foreach ($rawTables as $table) {
            if (!isset($table['data']) || !is_array($table['data'])) {
                continue;
            }
            $rows = [];
            foreach ($table['data'] as $row) {
                $cells = [];
                foreach ($row as $cell) {
                    $cells[] = isset($cell['text']) ? trim((string) $cell['text']) : '';
                }
                if (implode('', $cells) !== '') {
                    $rows[] = $cells;
                }
            }
            if (!empty($rows)) {
                $tables[] = $rows;
            }
        }
        return $tables;
    }

    /**
     * Convert extracted tables into shipments.
     */
    private function buildShipmentsFromTables(array $tables): array
    {
        $shipments = [];
        foreach ($tables as $rows) {
            if (!is_array($rows) || count($rows) === 0) { continue; }
            // Use headerless grouping uniformly to avoid missing multi-line shipments
            $colMap = $this->inferColumnIndices($rows);
            $shipments = array_merge($shipments, $this->buildShipmentsFromHeaderlessRows($rows, $colMap));
        }
        // Return all shipments as-is (no deduplication) per user requirement
        return $shipments;
    }

    /**
     * Simple row-wise mapping without filtering: create one shipment per data row.
     */
    private function buildShipmentsSimple(array $rows): array
    {
        $header = $rows[0];
        if (!is_array($header)) { return []; }

        $findIdx = function(array $hdr, array $needles, int $defaultIdx = -1): int {
            foreach ($hdr as $i => $label) {
                $norm = strtolower(preg_replace('/\s+/', '', (string)$label));
                foreach ($needles as $needle) {
                    $needleNorm = strtolower(preg_replace('/\s+/', '', $needle));
                    if (str_contains($norm, $needleNorm)) { return $i; }
                }
            }
            return $defaultIdx;
        };

        $carrierIdx = $findIdx($header, ['carrier','airbill','consignment','tracking','waybill','parcel']);
        $senderIdx = $findIdx($header, ['sender','from','sentfrom','shipper','senderaddress']);
        $receiverIdx = $findIdx($header, ['receiver','to','destination','dest','consignee','receiveraddress']);
        $piecesIdx = $findIdx($header, ['pieces','parcels','piecesweightdimensionszone','piecesweight']);
        $chargesIdx = $findIdx($header, ['charges','service']);
        $totalIdx = $findIdx($header, ['total','amount']);
        $refIdx = $findIdx($header, ['reference','ref']);

        $shipments = [];
        for ($r = 1; $r < count($rows); $r++) {
            $row = $rows[$r]; if (!is_array($row)) continue;
            $carrierCell = $carrierIdx >= 0 && isset($row[$carrierIdx]) ? (string)$row[$carrierIdx] : '';
            $senderCell = $senderIdx >= 0 && isset($row[$senderIdx]) ? (string)$row[$senderIdx] : '';
            $receiverCell = $receiverIdx >= 0 && isset($row[$receiverIdx]) ? (string)$row[$receiverIdx] : '';
            $piecesCell = $piecesIdx >= 0 && isset($row[$piecesIdx]) ? (string)$row[$piecesIdx] : '';
            $chargesCell = $chargesIdx >= 0 && isset($row[$chargesIdx]) ? (string)$row[$chargesIdx] : '';
            $totalCell = $totalIdx >= 0 && isset($row[$totalIdx]) ? (string)$row[$totalIdx] : '';
            $refCell = $refIdx >= 0 && isset($row[$refIdx]) ? (string)$row[$refIdx] : '';

            $parcel = null;
            if (preg_match('/\b\d{8,}\b/', $carrierCell, $m)) { $parcel = $m[0]; }
            $courier = null;
            if (preg_match('/(DHL|TNT|FedEx)/i', $carrierCell, $m)) { $courier = $m[0]; }

            [$fromName, $fromAddr] = $this->sanitizeNameAndAddressFromCell($senderCell);
            [$toName, $toAddr] = $this->sanitizeNameAndAddressFromCell($receiverCell);

            $noParcels = null; if (preg_match('/x(\d+)/i', $piecesCell, $m)) { $noParcels = $m[1]; }
            $weight = null; if (preg_match('/(\d+\.?\d*)\s*kg/i', $piecesCell, $m)) { $weight = $m[1]; }
            $total = $this->toNumber($totalCell);
            if ($total === null && strpos($chargesCell, '£') !== false) { $total = $this->toNumber($chargesCell); }
            $reference = trim($refCell) !== '' ? trim($refCell) : null;
            if (!$reference && preg_match('/\b(ref(?:erence)?|po\s*-?\s*\d+)\b/i', $chargesCell)) { $reference = trim($chargesCell); }

            $shipment = [
                'parcel_number' => $parcel,
                'courier' => $courier,
                'department' => $reference ? $this->deriveDepartmentFromReference($reference) : null,
                'reference' => $reference,
                'sent_from_name' => $fromName,
                'destination_name' => $toName,
                'sent_from_address' => $fromAddr,
                'destination_address' => $toAddr,
                'country' => null,
                'weight' => $weight,
                'no_parcels' => $noParcels,
                'vat_amount' => null,
                'total' => $total !== null ? number_format($total, 2, '.', '') : null,
            ];

            // Only drop completely empty rows
            $nonEmpty = array_filter([
                $parcel,$courier,$fromName,$toName,$fromAddr,$toAddr,$noParcels,$weight,$total,$reference
            ], function($v){ return !($v === null || $v === '' ); });
            if (count($nonEmpty) === 0) { continue; }

            $shipments[] = $shipment;
        }
        return $shipments;
    }

    private function toNumber(?string $value): ?float
    {
        if ($value === null) return null;
        $clean = preg_replace('/[^0-9\.\-]/', '', $value);
        if ($clean === '' || $clean === '-' || $clean === '.') return null;
        return (float)$clean;
    }

    private function extractVatFromCells(array $cells): ?float
    {
        foreach ($cells as $cell) {
            $cs = (string)$cell;
            if (preg_match('/VAT\s*:\s*£\s*(\d+\.?\d*)/i', $cs, $m)) {
                return (float)$m[1];
            }
            if (preg_match('/\bVAT\b.*?£\s*(\d+\.?\d*)/i', $cs, $m)) {
                return (float)$m[1];
            }
        }
        return null;
    }

    private function extractReferenceFromCarrierCell(string $cell): ?string
    {
        // Take the last non-empty line as reference
        $lines = array_values(array_filter(array_map('trim', preg_split('/\r?\n/', $cell))));
        if (empty($lines)) return null;
        return end($lines);
    }

    private function headerLooksLikeLabels(array $header): bool
    {
        $joined = strtolower(implode(' ', array_map('strval', $header)));
        // If header looks like values (company names, x1, £ amounts), treat as headerless
        if (preg_match('/\b£\b|\bx\d+\b|\d+\.\d+|^dhl parcel uk$/i', $joined)) {
            return false;
        }
        // If common label words exist, treat as labeled
        $labels = ['parcel','tracking','consignment','courier','carrier','service','reference','total','vat','weight','pieces','parcels','department'];
        foreach ($labels as $lab) {
            if (str_contains($joined, $lab)) return true;
        }
        // Default to headerless
        return false;
    }

    private function inferColumnIndices(array $rows): array
    {
        // Heuristic: pick columns by inspecting the first 10 data rows
        $maxCols = 0;
        foreach ($rows as $r) { $maxCols = max($maxCols, is_array($r)?count($r):0); }
        $scores = array_fill(0, $maxCols, ['parcel'=>0,'total'=>0,'pieces'=>0,'courier'=>0,'sender'=>0,'receiver'=>0,'reference'=>0]);
        $sample = array_slice($rows, 0, min(10, count($rows)));
        foreach ($sample as $r) {
            for ($i=0;$i<count($r);$i++) {
                $cell = (string)$r[$i];
                if (preg_match('/\b\d{8,}\b/', $cell)) $scores[$i]['parcel']++;
                if (strpos($cell, '£') !== false || preg_match('/^\d+\.\d{2}$/', trim($cell))) $scores[$i]['total']++;
                if (preg_match('/^x\d+$/i', trim($cell))) $scores[$i]['pieces']++;
                if (preg_match('/^(DHL|TNT|FedEx)/i', trim($cell))) $scores[$i]['courier']++;
                if (preg_match('/\b(From|Sender|ROBERT TODD|RTS)\b/i', $cell)) $scores[$i]['sender']++;
                if (preg_match('/\b(To|Receiver|Consignee|Boden|J P Boden|BELSTAFF)\b/i', $cell)) $scores[$i]['receiver']++;
                if (preg_match('/\b(ref|reference)\b/i', $cell)) $scores[$i]['reference']++;
            }
        }
        $pick = function(string $key) use ($scores) {
            $best = -1; $bestIdx = -1;
            foreach ($scores as $i=>$sc) { if ($sc[$key] > $best) { $best = $sc[$key]; $bestIdx = $i; } }
            return $bestIdx;
        };
        return [
            'parcel'=>$pick('parcel'),
            'total'=>$pick('total'),
            'pieces'=>$pick('pieces'),
            'courier'=>$pick('courier'),
            'sender'=>$pick('sender'),
            'receiver'=>$pick('receiver'),
            'reference'=>$pick('reference'),
        ];
    }

    private function buildShipmentsFromHeaderlessRows(array $rows, array $colMap): array
    {
        $shipments = [];
        $normalize = function ($row): array {
            $out = [];
            foreach ((array) $row as $cell) { $out[] = trim((string) $cell); }
            return $out;
        };

        $n = count($rows);
        for ($i = 0; $i < $n; $i++) {
            if (!is_array($rows[$i])) {
                continue;
            }
            $r = $normalize($rows[$i]);
            // Header row heuristic: has pieces marker xN and a money amount anywhere
            $hasPieces = false; $hasAmount = false;
            foreach ($r as $cellChk) {
                $c = (string) $cellChk;
                if (preg_match('/^x\d+$/i', trim($c))) { $hasPieces = true; }
                if (strpos($c, '£') !== false || preg_match('/\b\d+\.\d{2}\b/', $c)) { $hasAmount = true; }
            }
            $isHeader = $hasPieces && $hasAmount;
            if (!$isHeader) {
                continue;
            }

            // Prefer inferred sender/receiver columns if available
            $senderIdx = $colMap['sender'] ?? -1;
            $receiverIdx = $colMap['receiver'] ?? -1;
            $courier = null;
            foreach ($r as $cell) { if (preg_match('/^(DHL|TNT|FedEx)/i', (string)$cell)) { $courier = trim((string)$cell); break; } }
            $fromLines = [];
            if ($senderIdx >= 0 && isset($r[$senderIdx]) && $r[$senderIdx] !== '') $fromLines[] = $r[$senderIdx];
            $toLines = [];
            if ($receiverIdx >= 0 && isset($r[$receiverIdx]) && $r[$receiverIdx] !== '') $toLines[] = $r[$receiverIdx];
            $noParcels = null; foreach ($r as $cell) { if (preg_match('/^x\d+$/i', trim((string)$cell))) { $noParcels = preg_replace('/[^0-9]/','',(string)$cell); break; } }
            $total = null; foreach ($r as $cell) { if (strpos((string)$cell,'£')!==false || preg_match('/\b\d+\.\d{2}\b/',(string)$cell)) { $total = $this->toNumber((string)$cell); if ($total!==null) break; } }
            $parcelNumber = null;
            $weight = null;
            $reference = $this->extractReferenceFromCarrierCell($r[0] ?? '');
            $vat = $this->extractVatFromCells($r);

            // Consume continuation rows until next header
            $j = $i + 1;
            for (; $j < $n; $j++) {
                if (!is_array($rows[$j])) {
                    break;
                }
                $nr = $normalize($rows[$j]);
                $nHasPieces = false; $nHasAmount = false;
                foreach ($nr as $nc) {
                    if (preg_match('/^x\d+$/i', trim((string)$nc))) { $nHasPieces = true; }
                    if (strpos((string)$nc,'£') !== false || preg_match('/\b\d+\.\d{2}\b/', (string)$nc)) { $nHasAmount = true; }
                }
                $nextIsHeader = $nHasPieces && $nHasAmount;
                if ($nextIsHeader) {
                    break;
                }
                if ($senderIdx >= 0 && isset($nr[$senderIdx]) && $nr[$senderIdx] !== '') $fromLines[] = $nr[$senderIdx];
                if ($receiverIdx >= 0 && isset($nr[$receiverIdx]) && $nr[$receiverIdx] !== '') $toLines[] = $nr[$receiverIdx];
                // Capture reference text from any cell; update VAT if present
                if (!$reference) {
                    foreach ($nr as $cell) {
                        $cs = (string)$cell;
                        if (preg_match('/\b(ref(?:erence)?|po\s*-?\s*\d+)\b/i', $cs)) { $reference = trim($cs); break; }
                        if (preg_match('/^(Domestic|Next Day|Economy|Express)/i', $cs)) { $reference = trim($cs); }
                    }
                }
                if ($vat === null) { $vat = $this->extractVatFromCells($nr); }
                if ($parcelNumber === null) {
                    foreach ($nr as $cell) {
                        if (preg_match('/\b\d{8,}\b/', (string) $cell, $m)) {
                            $parcelNumber = $m[0];
                            break;
                        }
                    }
                }
                if ($weight === null) {
                    foreach ($nr as $cell) {
                        if (preg_match('/(\d+\.?\d*)\s*kg/i', (string) $cell, $m)) {
                            $weight = $m[1];
                            break;
                        }
                    }
                }
            }

            [$fromName, $fromAddr] = $this->sanitizeNameAndAddressFromCell(implode("\n", $fromLines));
            [$toName, $toAddr] = $this->sanitizeNameAndAddressFromCell(implode("\n", $toLines));

            $shipments[] = [
                'parcel_number' => $parcelNumber,
                'courier' => $courier,
                'department' => null,
                'reference' => isset($reference) ? $reference : null,
                'sent_from_name' => $fromName,
                'destination_name' => $toName,
                'sent_from_address' => $fromAddr,
                'destination_address' => $toAddr,
                'country' => null,
                'weight' => $weight,
                'no_parcels' => $noParcels,
                'vat_amount' => $vat !== null ? number_format((float)$vat, 2, '.', '') : null,
                'total' => $total !== null ? number_format($total, 2, '.', '') : null,
            ];

            $i = $j - 1;
        }
        return $shipments;
    }
    /**
     * Choose the table that most likely contains shipment rows.
     */
    private function chooseBestShipmentTable(array $tables): ?array
    {
        $best = null;
        $bestScore = -INF;
        foreach ($tables as $rows) {
            if (!is_array($rows) || count($rows) < 2) continue;
            $header = $rows[0];
            if (!is_array($header)) continue;

            // Score header labels
            $score = 0;
            $labels = array_map(function ($v) {
                return strtolower(preg_replace('/\s+/', '', (string)$v));
            }, $header);

            $incIfPresent = function(array $needles, int $points) use ($labels) {
                foreach ($needles as $n) {
                    $n2 = strtolower(preg_replace('/\s+/', '', $n));
                    foreach ($labels as $lab) {
                        if (str_contains($lab, $n2)) return $points;
                    }
                }
                return 0;
            };

            $score += $incIfPresent(['parcel','parcelnumber','consignment','tracking','waybill'], 3);
            $score += $incIfPresent(['courier','carrier','service'], 2);
            $score += $incIfPresent(['reference','ref'], 2);
            $score += $incIfPresent(['department','dept'], 1);
            $score += $incIfPresent(['weight','weight(kg)','kg'], 1);
            $score += $incIfPresent(['parcels','pieces','pkgs','qty'], 1);
            $score += $incIfPresent(['total','amount','charge','charges'], 2);
            $score += $incIfPresent(['vat','vatamount'], 1);

            // Penalize obvious non-shipment summary tables
            if ($incIfPresent(['invoicenumber','duedate','invoicedate','amountdue'], 1) > 0 && count($header) <= 4) {
                $score -= 4;
            }

            // Inspect a couple of data rows for patterns (£ amounts, long numbers)
            $dataSample = array_slice($rows, 1, 5);
            $hasPound = false; $hasLongNum = false;
            foreach ($dataSample as $dataRow) {
                foreach ((array)$dataRow as $cell) {
                    $cellStr = (string)$cell;
                    if (strpos($cellStr, '£') !== false) $hasPound = true;
                    if (preg_match('/\b\d{8,}\b/', $cellStr)) $hasLongNum = true;
                }
            }
            if ($hasPound) $score += 1;
            if ($hasLongNum) $score += 2;

            if ($score > $bestScore) {
                $bestScore = $score;
                $best = $rows;
            }
        }

        if ($best) {
            try { \Log::info('Chosen table header: ' . json_encode($best[0])); } catch (\Throwable $e) {}
        }
        return $best;
    }

    /**
     * Clean a multi-line or delimited cell to extract a display name and address.
     * Removes non-address noise like weights, dimensions, VAT, totals, and service rows.
     * Returns [name, address]
     */
    private function sanitizeNameAndAddressFromCell(string $cell): array
    {
        // Normalize line breaks
        $raw = str_replace(["\r"], "\n", $cell);
        $raw = preg_replace('/\n{2,}/', "\n", $raw);
        $parts = array_values(array_filter(array_map('trim', preg_split('/\n|\|/', $raw))));

        $cleaned = [];
        foreach ($parts as $p) {
            $pNorm = trim($p);
            if ($pNorm === '') continue;
            // Drop obvious non-address/service lines
            if (preg_match('/^(x\d+|\d+\.\d*\s*kg|\d+\.\d*kg)$/i', $pNorm)) continue; // parcels/weight
            if (preg_match('/^\d+\.\d+x\d+\.\d+x\d+\.\d+$/i', $pNorm)) continue; // dimensions
            // Service lines (match anywhere in the line)
            if (preg_match('/(Economy|Express|Package|Fuel|Security|Brexit|Enhanced|Service|Zone|Charges|Reference Summary)/i', $pNorm)) continue;
            // Money / VAT lines and bare amounts
            if (preg_match('/^VAT\s*:|^£/i', $pNorm)) continue;
            if (preg_match('/^\d+(?:\.\d+)?$/', $pNorm)) continue;
            // Single-letter noise
            if (preg_match('/^[A-Z]$/', $pNorm)) continue;
            $cleaned[] = $pNorm;
        }

        // Merge common split country tokens (e.g., "The" + "Netherlands")
        $merged = [];
        for ($i = 0; $i < count($cleaned); $i++) {
            $line = $cleaned[$i];
            if (strcasecmp($line, 'The') === 0 && isset($cleaned[$i+1]) && strcasecmp($cleaned[$i+1], 'Netherlands') === 0) {
                $merged[] = 'The Netherlands';
                $i++; // skip next
                continue;
            }
            $merged[] = $line;
        }
        $cleaned = $merged;

        if (empty($cleaned)) {
            return [null, null];
        }

        // Identify likely country suffix to keep in address
        $countryHints = ['United Kingdom','UK','Netherlands','Romania','Bulgaria','China','Turkey','Hong Kong','India','Bangladesh'];
        // Create name as first line unless it contains postal code-like digits only
        $name = $cleaned[0];
        $addressLines = array_slice($cleaned, 1);

        // If first line looks not like a name but like an address continuation, try next
        if (preg_match('/^(x\d+|\d|\d+\s|\d{3,}.*|[A-Z]{2,}\s*\d)/', $name)) {
            // swap: push into address
            array_unshift($addressLines, $name);
            $name = isset($cleaned[1]) ? $cleaned[1] : $name;
            $addressLines = array_slice($cleaned, 2);
        }

        // Rebuild address preserving original line intent
        $address = implode("\n", $addressLines);
        if ($address === '') $address = null;
        if ($name === '') $name = null;
        return [$name, $address];
    }
    private function logTabulaPreview(array $tables): void
    {
        try {
            if (empty($tables)) {
                \Log::info('Tabula preview: no tables');
                return;
            }
            $first = $tables[0];
            $header = $first[0] ?? [];
            \Log::info('Tabula header: ' . json_encode($header));
            $sample = [];
            for ($i = 1; $i < min(4, count($first)); $i++) {
                $sample[] = $first[$i];
            }
            \Log::info('Tabula first rows: ' . json_encode($sample));
        } catch (\Throwable $e) {
            // ignore logging failures
        }
    }

    private function extractTextFromPdf($filePath)
    {
        // Verify file exists and is readable
        if (!file_exists($filePath) || !is_readable($filePath)) {
            throw new \Exception("PDF file not found or not readable: $filePath");
        }

        // First try using shell command (pdftotext) - faster and more reliable
        if (function_exists('shell_exec')) {
            $command = "pdftotext -q \"$filePath\" - 2>&1";
            $output = shell_exec($command);
            
            if ($output !== null && !empty(trim($output))) {
                return "=== PDF TEXT EXTRACTED (pdftotext) ===\n\n" . $output;
            }
        }
        
        // Fallback: use PHP PDF parser library
        try {
            $parser = new Parser();
            $pdf = $parser->parseFile($filePath);
            $text = $pdf->getText();
            
            if (!empty(trim($text))) {
                return "=== PDF TEXT EXTRACTED (PHP Parser) ===\n\n" . $text;
            }
        } catch (\Exception $e) {
            // If both methods fail, throw exception
            throw new \Exception("PHP PDF parser failed: " . $e->getMessage());
        }
        
        throw new \Exception("PDF text extraction failed. The PDF might be empty or corrupted.");
    }

    private function parseInvoiceData($text)
    {
        $shipments = [];
        $lines = explode("\n", $text);
        
        \Log::info("Total lines in PDF: " . count($lines));
        
        // Department mapping with typo handling
        $departmentMap = [
            'KWA' => 'KWA',
            'KWT' => 'KWT',
            'KTW' => 'KWT', // Typo handling
            'tom' => 'KWT', // Typo handling
            'KCE' => 'KCE',
            'YS' => 'YS',
            'Yarn Store' => 'YS',
            'YC' => 'YC',
            'Yarn' => 'YC',
            'OH' => 'OH'
        ];
        
        $currentShipment = null;
        $seenParcelNumbers = [];
        $lineIndex = 0;
        
        foreach ($lines as $lineIndex => $line) {
            // Normalize leading unicode spaces/controls then trim
            $line = preg_replace('/^[\x{2000}-\x{200B}\x{FEFF}\pC]+/u', '', $line);
            $line = trim($line);
            
            // Look for carrier lines (DHL, TNT, FedEx) - multiple patterns
            $shipmentFound = false;
            $courier = null;
            $parcelNumber = null;
            
            // Pattern 1: Standard format (DHL - 1234567890)
            if (preg_match('/^(DHL|TNT|FedEx)\s*[-–]\s*(\d+)/', $line, $matches)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 2: Domestic/Express format (DHL Domestic 1234567890)
            elseif (preg_match('/^(DHL|TNT|FedEx)\s+(?:Domestic|Express|Parcel|Worldwide)\s+(\d+)/', $line, $matches)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 3: Courier + parcel number anywhere on the line (exclude pure service header lines)
            elseif (preg_match('/(DHL|TNT|FedEx)[^\d\n]*?(\d{8,})/', $line, $matches) && !preg_match('/^(DHL|TNT|FedEx)\s+(?:Parcel|Express|Domestic|Worldwide)$/', $line)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 4: Two-line format - courier name only (look ahead up to 5 lines for parcel number)
            elseif (preg_match('/^(DHL|TNT|FedEx)\s+(?:Parcel|Express|Domestic|Worldwide)/', $line, $matches)) {
                $courier = $matches[1];
                // Look ahead a few lines for parcel number to handle line breaks/extra service lines
                for ($lookAhead = 1; $lookAhead <= 5; $lookAhead++) {
                    if ($lineIndex + $lookAhead < count($lines)) {
                        $nextLine = trim($lines[$lineIndex + $lookAhead]);
                        if (preg_match('/(?:Domestic|Express|Parcel|Worldwide)\s+(\d{8,})/', $nextLine, $nextMatches)) {
                            $shipmentFound = true;
                            $parcelNumber = $nextMatches[1];
                            \Log::info("Found two-line shipment: {$courier} - {$parcelNumber} (courier: '$line', parcel: '$nextLine')");
                            break;
                        }
                    }
                }
            }
            // Pattern 4b: Standalone service+parcel line (e.g., "Domestic 60120209842409") with context search
            elseif (preg_match('/^(?:Domestic|Express|Parcel|Worldwide)\s+(\d{8,})$/', $line, $matches)) {
                $parcelNumber = $matches[1];
                $courier = null;
                // Look back up to 20 lines to find courier context or 'DHL Parcel UK'
                for ($back = 1; $back <= 20; $back++) {
                    if ($lineIndex - $back >= 0) {
                        $prev = trim($lines[$lineIndex - $back]);
                        if (preg_match('/^(DHL|TNT|FedEx)/', $prev, $m) || stripos($prev, 'DHL Parcel UK') !== false) {
                            $courier = isset($m[1]) ? $m[1] : 'DHL';
                            break;
                        }
                    }
                }
                // If still not found, try forward a few lines
                if (!$courier) {
                    for ($fwd = 1; $fwd <= 5; $fwd++) {
                        if ($lineIndex + $fwd < count($lines)) {
                            $nxt = trim($lines[$lineIndex + $fwd]);
                            if (preg_match('/^(DHL|TNT|FedEx)/', $nxt, $m) || stripos($nxt, 'DHL Parcel UK') !== false) {
                                $courier = isset($m[1]) ? $m[1] : 'DHL';
                                break;
                            }
                        }
                    }
                }
                // If still no explicit courier context, default to DHL for this invoice format
                if (!$courier) {
                    $courier = 'DHL';
                }
                $shipmentFound = true;
            }
            // Pattern 5: New shipment detection after total lines - this is the key pattern
            elseif (preg_match('/^(DHL|TNT|FedEx)/', $line, $matches)) {
                // Look back to find the most recent total line (end of previous shipment)
                $foundTotal = false;
                $totalLineIndex = -1;
                
                for ($i = $lineIndex - 1; $i >= max(0, $lineIndex - 20); $i--) {
                    $checkLine = trim($lines[$i]);
                    // Look for total lines that end with £ amount
                    if (preg_match('/£\s*\d+\.?\d*\s*$/', $checkLine)) {
                        $foundTotal = true;
                        $totalLineIndex = $i;
                        break;
                    }
                }
                
                if ($foundTotal) {
                    $courier = $matches[1];
                    // Look ahead up to 3 lines for parcel number (skip service description lines)
                    for ($lookAhead = 1; $lookAhead <= 3; $lookAhead++) {
                        if ($lineIndex + $lookAhead < count($lines)) {
                            $nextLine = trim($lines[$lineIndex + $lookAhead]);
                            if (preg_match('/(\d{8,})/', $nextLine, $nextMatches)) {
                                $parcelNumber = $nextMatches[1];
                                // Check if already processed
                                $alreadyProcessed = false;
                                foreach ($shipments as $existingShipment) {
                                    if ($existingShipment['parcel_number'] === $parcelNumber) {
                                        $alreadyProcessed = true;
                                        break;
                                    }
                                }
                                if (!$alreadyProcessed) {
                                    $shipmentFound = true;
                                    \Log::info("Found post-total shipment: {$courier} - {$parcelNumber} (after total line {$totalLineIndex}, courier: '$line', parcel: '$nextLine')");
                                }
                                break; // stop after first parcel number found
                            }
                        }
                    }
                }
            }
            
            // Prevent duplicates using a fast set of already-seen parcel numbers
            if ($shipmentFound && $parcelNumber) {
                if (isset($seenParcelNumbers[$parcelNumber])) {
                    \Log::info("Duplicate shipment detected, skipping: {$parcelNumber}");
                    $shipmentFound = false;
                } else {
                    $seenParcelNumbers[$parcelNumber] = true;
                }
            }
            
            if ($shipmentFound) {
                \Log::info("Found shipment: {$courier} - {$parcelNumber} in line: '$line'");
                if ($currentShipment) {
                    // Collapse collected references into a single string on the shipment being closed
                    if (isset($currentShipment['references']) && is_array($currentShipment['references'])) {
                        $collapsed = trim(implode(' | ', array_unique(array_map('trim', $currentShipment['references']))));
                        $currentShipment['reference'] = $collapsed;
                        // Derive department from reference if not set
                        if (empty($currentShipment['department'])) {
                            $currentShipment['department'] = $this->deriveDepartmentFromReference($collapsed);
                        }
                        unset($currentShipment['references'], $currentShipment['ref_open'], $currentShipment['await_total']);
                    }
                    $shipments[] = $currentShipment;
                }
                
                $currentShipment = [
                    'parcel_number' => $parcelNumber,
                    'courier' => $courier,
                    'department' => null,
                    'weight' => null,
                    'no_parcels' => null,
                    'destination_name' => null,
                    'sent_from_name' => null,
                    'sent_from_address' => null,
                    'destination_address' => null,
                    'country' => null,
                    'vat_amount' => null,
                    'total' => null,
                    'references' => [],
                    'ref_open' => true,
                    'await_total' => null
                ];
                
                // Look ahead for department and addresses
                $this->extractDepartmentAndAddresses($currentShipment, $lines, $lineIndex, $departmentMap);
            }
            // Debug: Log lines that contain courier names but don't match any pattern
            elseif (strpos($line, 'DHL') !== false || strpos($line, 'TNT') !== false || strpos($line, 'FedEx') !== false) {
                \Log::info("Line contains courier but no match: '$line'");
            }
            
            // Debug: Log lines that contain parcel numbers to understand the structure
            if (preg_match('/(\d{8,})/', $line, $matches)) {
                \Log::info("Line contains parcel number: '{$matches[1]}' in line: '$line'");
            }

            // Capture reference-like lines only while ref window is open
            if ($currentShipment && !empty($currentShipment['ref_open'])) {
                $refLine = $line;
                // Heuristics: exclude obvious non-reference noise
                $isNoise = preg_match('/^(GB\s*\/\s*[A-Z]{2})$/', $refLine)
                    || preg_match('/^\d{2}\/\d{2}\/\d{4}$/', $refLine)
                    || preg_match('/^(x\d+|\d+\.\d*kg|\d+\.\d*x\d+\.\d*x\d+\.\d+|VAT\s*:|£)/i', $refLine)
                    || preg_match('/^(DHL|TNT|FedEx)/i', $refLine)
                    || preg_match('/^(Reference Summary|Reference\s*\d+|Pieces|Weight|Dimensions|Zone|Charges|Customer|Invoice|Sender Address|Receiver Address)$/i', $refLine)
                    || $refLine === '';
                if (!$isNoise) {
                    // Reference if dept code present OR product-type keywords present
                    if (preg_match('/\b(KWA|KWT|KTW|KCE|YS|YC|OH)\b/i', $refLine)
                        || preg_match('/\b(button|yarn|pcs|reference|ref|bulk)\b/i', $refLine)
                        || preg_match('/\b(erdos)\b/i', $refLine)) {
                        $currentShipment['references'][] = trim($refLine);
                    }
                }
            }

            // Extract weight (look for kg)
            if ($currentShipment && preg_match('/(\d+\.?\d*)kg/', $line, $matches)) {
                $currentShipment['weight'] = $matches[1];
            }
            
            // Alternative weight extraction (look for weight patterns in lines with parcel counts)
            if ($currentShipment && preg_match('/x\d+.*?(\d+\.?\d*)kg/', $line, $matches)) {
                $currentShipment['weight'] = $matches[1];
                // Also try to extract parcel count from the same line
                if (preg_match('/x(\d+)/', $line, $parcelMatches)) {
                    $currentShipment['no_parcels'] = $parcelMatches[1];
                }
            }
            
            // Extract number of parcels (look for x1, x2, etc.) - but NOT dimension patterns like 0.00x0.00x0.00
            if ($currentShipment && preg_match('/x(\d+)/', $line, $matches) && !preg_match('/^\d+\.\d+x\d+\.\d+x\d+\.\d+$/', $line)) {
                $currentShipment['no_parcels'] = $matches[1];
                \Log::info("Found parcel count '{$matches[1]}' in line: '$line' for shipment {$currentShipment['parcel_number']}");
            }
            
            // Alternative parcel count extraction (look for patterns like "x1," or "x1 " at start of line)
            if ($currentShipment && preg_match('/^x(\d+)/', $line, $matches)) {
                $currentShipment['no_parcels'] = $matches[1];
                \Log::info("Found start-of-line parcel count '{$matches[1]}' in line: '$line' for shipment {$currentShipment['parcel_number']}");
            }
            
            // Debug: Log lines that might contain parcel information
            if ($currentShipment && (strpos($line, 'x') !== false || strpos($line, 'kg') !== false)) {
                \Log::info("Potential parcel/weight line for shipment {$currentShipment['parcel_number']}: '$line'");
            }
            
            // On VAT line, capture VAT and open a short window to accept the next standalone total
            if ($currentShipment && preg_match('/VAT\s*:\s*£\s*(\d+\.?\d*)/i', $line, $matches)) {
                $currentShipment['vat_amount'] = $matches[1];
                $currentShipment['ref_open'] = false;
                $currentShipment['await_total'] = 6; // look for total within next ~6 lines
            }

            // Totals: only accept standalone £ lines while await_total window is open, and ignore doc-level totals
            if ($currentShipment && isset($currentShipment['await_total']) && $currentShipment['await_total'] > 0) {
                if (preg_match('/^\s*£\s*(\d+(?:\.\d{2})?)\s*$/', $line, $mTot)) {
                    $prev1Raw = trim($lines[$lineIndex - 1] ?? '');
                    $isDoc = preg_match('/(customer total|totals:|amount due|paid\/credited|remaining|vat summary|reference summary)/i', strtolower($prev1Raw . ' ' . $line));
                    if (!$isDoc) { $currentShipment['total'] = $mTot[1]; $currentShipment['await_total'] = 0; }
                } else {
                    // decrement window until consumed
                    $currentShipment['await_total'] = max(0, (int) $currentShipment['await_total'] - 1);
                }
            }
        }
        
        // Add the last shipment
        if ($currentShipment) {
            if (isset($currentShipment['references']) && is_array($currentShipment['references'])) {
                $collapsed = trim(implode(' | ', array_unique(array_map('trim', $currentShipment['references']))));
                $currentShipment['reference'] = $collapsed;
                if (empty($currentShipment['department'])) {
                    $currentShipment['department'] = $this->deriveDepartmentFromReference($collapsed);
                }
                unset($currentShipment['references'], $currentShipment['ref_open'], $currentShipment['await_total']);
            }
            $shipments[] = $currentShipment;
        }
        
        return $shipments;
    }

    private function extractDepartmentAndAddresses(&$shipment, $lines, $startIndex, $departmentMap)
    {
        $senderAddress = [];
        $receiverAddress = [];
        $foundSender = false;
        $foundReceiver = false;
        
        // Look ahead up to 50 lines for all information
        for ($i = $startIndex + 1; $i < min($startIndex + 50, count($lines)); $i++) {
            $line = trim($lines[$i]);
            
            // Skip empty lines
            if (empty($line)) continue;
            
            // Look for the next shipment to stop processing
            if (preg_match('/^(DHL|TNT|FedEx).*?(\d+)/', $line)) {
                break;
            }
            
            // Look for department in reference lines (more comprehensive)
            if (strpos($line, 'Reference') !== false) {
                foreach ($departmentMap as $code => $mappedCode) {
                    if (stripos($line, $code) !== false) {
                        $shipment['department'] = $mappedCode;
                        break;
                    }
                }
                continue;
            }
            
            // Look for pieces/weight line to identify end of address section
            if (preg_match('/x\d+.*kg/', $line)) {
                break;
            }
            
            // Look for GB / CN, GB / RO patterns (origin/destination) - skip these
            if (preg_match('/GB\s*\/\s*([A-Z]{2})/', $line, $matches)) {
                continue;
            }
            
            // Look for customer number lines - skip these
            if (preg_match('/^\d{8,}/', $line)) {
                continue;
            }
            
            // Look for date patterns - skip these
            if (preg_match('/^\d{2}\/\d{2}\/\d{4}/', $line)) {
                continue;
            }
            
            // Look for ROBERT TODD SONS to identify sender section
            if (stripos($line, 'ROBERT TODD SONS') !== false) {
                $foundSender = true;
                $shipment['sent_from_name'] = 'ROBERT TODD SONS';
                $senderAddress[] = $line;
                continue;
            }
            
            // Look for NARROW LANE, WYMESWOLD, LOUGHBOROUGH to continue sender address
            if ($foundSender && (stripos($line, 'NARROW LANE') !== false || stripos($line, 'WYMESWOLD') !== false || stripos($line, 'LOUGHBOROUGH') !== false || stripos($line, 'LE12 6SD') !== false)) {
                $senderAddress[] = $line;
                continue;
            }
            
            // Look for country patterns to identify receiver address
            if (preg_match('/(United Kingdom|China|Romania|Bulgaria|Turkey|Hong Kong|India|Bangladesh)/', $line, $matches)) {
                if (!$foundReceiver) {
                    $receiverAddress[] = $line;
                    $foundReceiver = true;
                    $shipment['country'] = $matches[1];
                }
                continue;
            }
            
            // Look for receiver name (first line that's not sender-related and not a country)
            if (!$foundReceiver && !preg_match('/^(DHL|TNT|FedEx|Reference|Pieces|Weight|Charges|Customer|Invoice|GB\/|ROBERT|NARROW|WYMESWOLD|LOUGHBOROUGH|LE12|United Kingdom|China|Romania|Bulgaria|Turkey|Hong Kong|India|Bangladesh)/', $line)) {
                // Avoid treating reference/service lines as destination name
                $isReferenceLike = preg_match('/\b(KWA|KWT|KTW|KCE|YS|YC|OH)\b/i', $line)
                    || preg_match('/\b(button|yarn|pcs|reference|ref|bulk)\b/i', $line)
                    || preg_match('/\b(erdos|oliver|celtic|bonas|upw)\b/i', $line)
                    || preg_match('/^Domestic\s+\d{8,}$/i', $line)
                    || stripos($line, 'Next Day') === 0;
                if (!$isReferenceLike) {
                    if (empty($shipment['destination_name'])) {
                        $shipment['destination_name'] = $line;
                    } else {
                        $receiverAddress[] = $line;
                    }
                }
                continue;
            }
            // Continue building receiver address (but only if we've found the receiver name)
            if ($foundReceiver && !preg_match('/^(DHL|TNT|FedEx|Reference|Pieces|Weight|Charges|Customer|Invoice|GB\/|ROBERT|NARROW|WYMESWOLD|LOUGHBOROUGH|LE12|x\d+|kg|£|VAT|Package|Fuel|Brexit|Enhanced|Security|Supplement|Economy|Express|G|◀|▶)/', $line)) {
                // Skip reference/service lines inside destination address
                $isReferenceLike = preg_match('/\b(KWA|KWT|KTW|KCE|YS|YC|OH)\b/i', $line)
                    || preg_match('/\b(button|yarn|pcs|reference|ref|bulk)\b/i', $line)
                    || preg_match('/\b(erdos|oliver|celtic|bonas|upw)\b/i', $line)
                    || preg_match('/^Domestic\s+\d{8,}$/i', $line)
                    || stripos($line, 'Next Day') === 0;
                if ($isReferenceLike) {
                    continue;
                }
                // Stop collecting address if we hit weight or package/service information
                if (preg_match('/(\d+\.\d*)kg|Package|£|x\d+\.\d*x\d+\.\d*x\d+\.\d*|Economy|Express|G|◀|▶/', $line)) {
                    break;
                }
                $receiverAddress[] = $line;
                continue;
            }
        }
        
        $shipment['sent_from_address'] = implode(', ', array_filter($senderAddress));
        $shipment['destination_address'] = implode(', ', array_filter($receiverAddress));
    }

    private function deriveDepartmentFromReference(?string $reference): ?string
    {
        if (!$reference) {
            return null;
        }
        $ref = mb_strtolower($reference);
        // KCE
        if (preg_match('/\bkce\b/i', $reference) || preg_match('/\berdos\b/i', $reference) || preg_match('/^er\b/i', $reference)) {
            return 'KCE';
        }
        // KWA
        if (preg_match('/\bkaw\b/i', $reference) || preg_match('/\bkwa\b/i', $reference) || preg_match('/\bwholesale amy\b/i', $reference) || preg_match('/\bamy\b/i', $reference)) {
            return 'KWA';
        }
        // KWT
        if (preg_match('/\bkwt\b/i', $reference) || preg_match('/\bktw\b/i', $reference) || preg_match('/\btom\b/i', $reference) || preg_match('/\bwholesale\b/i', $reference) || preg_match('/\brt\b/i', $reference)) {
            return 'KWT';
        }
        // OH
        if (preg_match('/\boh\b/i', $reference) || preg_match('/\boverheads\b/i', $reference)) {
            return 'OH';
        }
        // YS
        if (preg_match('/\bys\b/i', $reference) || preg_match('/\byarn store\b/i', $reference)) {
            return 'YS';
        }
        // YC
        if (preg_match('/\byc\b/i', $reference) || preg_match('/\byarn\b/i', $reference) || preg_match('/\bsue\b/i', $reference)) {
            return 'YC';
        }
        return null;
    }

    public function render()
    {
        // Debug: Log that component is rendering
        \Log::info('ProcessInxpressInvoice component rendering');
        
        return view('livewire.process-inxpress-invoice');
    }
}
