<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Rule;
use Smalot\PdfParser\Parser;

class ProcessInxpressInvoice extends Component
{
    use WithFileUploads;

    #[Rule('required|file|mimes:pdf|max:10240')] // 10MB max
    public $pdfFile;

    public function processInvoice()
    {
        $this->validate();

        if ($this->pdfFile) {
            try {
                // Store the file temporarily
                $path = $this->pdfFile->store('temp');
                
                // Get the full path - store() returns relative path from storage/app/
                $fullPath = storage_path('app/' . $path);
                
                // Debug: Log the paths to see what's happening
                \Log::info("Stored path: " . $path);
                \Log::info("Full path: " . $fullPath);
                \Log::info("File exists: " . (file_exists($fullPath) ? 'YES' : 'NO'));
                
                // Alternative: try using the temporary path directly
                if (!file_exists($fullPath)) {
                    $altPath = $this->pdfFile->getRealPath();
                    \Log::info("Alternative path: " . $altPath);
                    if ($altPath && file_exists($altPath)) {
                        $fullPath = $altPath;
                        \Log::info("Using alternative path: " . $fullPath);
                    }
                }
                
                // Extract text from PDF
                $text = $this->extractTextFromPdf($fullPath);
                
                // Clean up the temporary file
                if (file_exists($fullPath)) {
                    unlink($fullPath);
                }
                
                // Parse the PDF text and extract shipment data
                $shipments = $this->parseInvoiceData($text);
                
                // Dump and die with the parsed data
                dd($shipments);
                
            } catch (\Exception $e) {
                // Clean up the temporary file if it exists
                if (isset($fullPath) && file_exists($fullPath)) {
                    unlink($fullPath);
                }
                
                // Show error
                dd("Error processing PDF: " . $e->getMessage());
            }
        }
    }

    private function extractTextFromPdf($filePath)
    {
        // Verify file exists and is readable
        if (!file_exists($filePath) || !is_readable($filePath)) {
            throw new \Exception("PDF file not found or not readable: $filePath");
        }

        // First try using shell command (pdftotext) - faster and more reliable
        if (function_exists('shell_exec')) {
            $command = "pdftotext -q \"$filePath\" - 2>&1";
            $output = shell_exec($command);
            
            if ($output !== null && !empty(trim($output))) {
                return "=== PDF TEXT EXTRACTED (pdftotext) ===\n\n" . $output;
            }
        }
        
        // Fallback: use PHP PDF parser library
        try {
            $parser = new Parser();
            $pdf = $parser->parseFile($filePath);
            $text = $pdf->getText();
            
            if (!empty(trim($text))) {
                return "=== PDF TEXT EXTRACTED (PHP Parser) ===\n\n" . $text;
            }
        } catch (\Exception $e) {
            // If both methods fail, throw exception
            throw new \Exception("PHP PDF parser failed: " . $e->getMessage());
        }
        
        throw new \Exception("PDF text extraction failed. The PDF might be empty or corrupted.");
    }

    private function parseInvoiceData($text)
    {
        $shipments = [];
        $lines = explode("\n", $text);
        
        \Log::info("Total lines in PDF: " . count($lines));
        
        // Department mapping with typo handling
        $departmentMap = [
            'KWA' => 'KWA',
            'KWT' => 'KWT',
            'KTW' => 'KWT', // Typo handling
            'KCE' => 'KCE',
            'YS' => 'YS',
            'YC' => 'YC',
            'OH' => 'OH'
        ];
        
        $currentShipment = null;
        $seenParcelNumbers = [];
        $lineIndex = 0;
        
        foreach ($lines as $lineIndex => $line) {
            $line = trim($line);
            
            // Look for carrier lines (DHL, TNT, FedEx) - multiple patterns
            $shipmentFound = false;
            $courier = null;
            $parcelNumber = null;
            
            // Pattern 1: Standard format (DHL - 1234567890)
            if (preg_match('/^(DHL|TNT|FedEx)\s*[-–]\s*(\d+)/', $line, $matches)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 2: Domestic/Express format (DHL Domestic 1234567890)
            elseif (preg_match('/^(DHL|TNT|FedEx)\s+(?:Domestic|Express|Parcel|Worldwide)\s+(\d+)/', $line, $matches)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 3: Any line starting with courier + parcel number (but exclude two-line format)
            elseif (preg_match('/^(DHL|TNT|FedEx).*?(\d{8,})/', $line, $matches) && !preg_match('/^(DHL|TNT|FedEx)\s+(?:Parcel|Express|Domestic|Worldwide)$/', $line)) {
                $shipmentFound = true;
                $courier = $matches[1];
                $parcelNumber = $matches[2];
            }
            // Pattern 4: Two-line format - courier name only (look ahead up to 5 lines for parcel number)
            elseif (preg_match('/^(DHL|TNT|FedEx)\s+(?:Parcel|Express|Domestic|Worldwide)/', $line, $matches)) {
                $courier = $matches[1];
                // Look ahead a few lines for parcel number to handle line breaks/extra service lines
                for ($lookAhead = 1; $lookAhead <= 5; $lookAhead++) {
                    if ($lineIndex + $lookAhead < count($lines)) {
                        $nextLine = trim($lines[$lineIndex + $lookAhead]);
                        if (preg_match('/(?:Domestic|Express|Parcel|Worldwide)\s+(\d{8,})/', $nextLine, $nextMatches)) {
                            $shipmentFound = true;
                            $parcelNumber = $nextMatches[1];
                            \Log::info("Found two-line shipment: {$courier} - {$parcelNumber} (courier: '$line', parcel: '$nextLine')");
                            break;
                        }
                    }
                }
            }
            // Pattern 4b: Standalone service+parcel line (e.g., "Domestic 60120209842409") with context search
            elseif (preg_match('/^(?:Domestic|Express|Parcel|Worldwide)\s+(\d{8,})$/', $line, $matches)) {
                $parcelNumber = $matches[1];
                $courier = null;
                // Look back up to 20 lines to find courier context or 'DHL Parcel UK'
                for ($back = 1; $back <= 20; $back++) {
                    if ($lineIndex - $back >= 0) {
                        $prev = trim($lines[$lineIndex - $back]);
                        if (preg_match('/^(DHL|TNT|FedEx)/', $prev, $m) || stripos($prev, 'DHL Parcel UK') !== false) {
                            $courier = isset($m[1]) ? $m[1] : 'DHL';
                            break;
                        }
                    }
                }
                // If still not found, try forward a few lines
                if (!$courier) {
                    for ($fwd = 1; $fwd <= 5; $fwd++) {
                        if ($lineIndex + $fwd < count($lines)) {
                            $nxt = trim($lines[$lineIndex + $fwd]);
                            if (preg_match('/^(DHL|TNT|FedEx)/', $nxt, $m) || stripos($nxt, 'DHL Parcel UK') !== false) {
                                $courier = isset($m[1]) ? $m[1] : 'DHL';
                                break;
                            }
                        }
                    }
                }
                // If still no explicit courier context, default to DHL for this invoice format
                if (!$courier) {
                    $courier = 'DHL';
                }
                $shipmentFound = true;
            }
            // Pattern 5: New shipment detection after total lines - this is the key pattern
            elseif (preg_match('/^(DHL|TNT|FedEx)/', $line, $matches)) {
                // Look back to find the most recent total line (end of previous shipment)
                $foundTotal = false;
                $totalLineIndex = -1;
                
                for ($i = $lineIndex - 1; $i >= max(0, $lineIndex - 20); $i--) {
                    $checkLine = trim($lines[$i]);
                    // Look for total lines that end with £ amount
                    if (preg_match('/£\s*\d+\.?\d*\s*$/', $checkLine)) {
                        $foundTotal = true;
                        $totalLineIndex = $i;
                        break;
                    }
                }
                
                if ($foundTotal) {
                    $courier = $matches[1];
                    // Look ahead up to 3 lines for parcel number (skip service description lines)
                    for ($lookAhead = 1; $lookAhead <= 3; $lookAhead++) {
                        if ($lineIndex + $lookAhead < count($lines)) {
                            $nextLine = trim($lines[$lineIndex + $lookAhead]);
                            if (preg_match('/(\d{8,})/', $nextLine, $nextMatches)) {
                                $parcelNumber = $nextMatches[1];
                                // Check if already processed
                                $alreadyProcessed = false;
                                foreach ($shipments as $existingShipment) {
                                    if ($existingShipment['parcel_number'] === $parcelNumber) {
                                        $alreadyProcessed = true;
                                        break;
                                    }
                                }
                                if (!$alreadyProcessed) {
                                    $shipmentFound = true;
                                    \Log::info("Found post-total shipment: {$courier} - {$parcelNumber} (after total line {$totalLineIndex}, courier: '$line', parcel: '$nextLine')");
                                }
                                break; // stop after first parcel number found
                            }
                        }
                    }
                }
            }
            
            // Prevent duplicates using a fast set of already-seen parcel numbers
            if ($shipmentFound && $parcelNumber) {
                if (isset($seenParcelNumbers[$parcelNumber])) {
                    \Log::info("Duplicate shipment detected, skipping: {$parcelNumber}");
                    $shipmentFound = false;
                } else {
                    $seenParcelNumbers[$parcelNumber] = true;
                }
            }
            
            if ($shipmentFound) {
                \Log::info("Found shipment: {$courier} - {$parcelNumber} in line: '$line'");
                if ($currentShipment) {
                    $shipments[] = $currentShipment;
                }
                
                $currentShipment = [
                    'parcel_number' => $parcelNumber,
                    'courier' => $courier,
                    'department' => null,
                    'weight' => null,
                    'no_parcels' => null,
                    'destination_name' => null,
                    'sent_from_name' => null,
                    'sent_from_address' => null,
                    'destination_address' => null,
                    'country' => null,
                    'vat_amount' => null,
                    'total' => null
                ];
                
                // Look ahead for department and addresses
                $this->extractDepartmentAndAddresses($currentShipment, $lines, $lineIndex, $departmentMap);
            }
            // Debug: Log lines that contain courier names but don't match any pattern
            elseif (strpos($line, 'DHL') !== false || strpos($line, 'TNT') !== false || strpos($line, 'FedEx') !== false) {
                \Log::info("Line contains courier but no match: '$line'");
            }
            
            // Debug: Log lines that contain parcel numbers to understand the structure
            if (preg_match('/(\d{8,})/', $line, $matches)) {
                \Log::info("Line contains parcel number: '{$matches[1]}' in line: '$line'");
            }
            
            // Extract weight (look for kg)
            if ($currentShipment && preg_match('/(\d+\.?\d*)kg/', $line, $matches)) {
                $currentShipment['weight'] = $matches[1];
            }
            
            // Alternative weight extraction (look for weight patterns in lines with parcel counts)
            if ($currentShipment && preg_match('/x\d+.*?(\d+\.?\d*)kg/', $line, $matches)) {
                $currentShipment['weight'] = $matches[1];
                // Also try to extract parcel count from the same line
                if (preg_match('/x(\d+)/', $line, $parcelMatches)) {
                    $currentShipment['no_parcels'] = $parcelMatches[1];
                }
            }
            
            // Extract number of parcels (look for x1, x2, etc.) - but NOT dimension patterns like 0.00x0.00x0.00
            if ($currentShipment && preg_match('/x(\d+)/', $line, $matches) && !preg_match('/^\d+\.\d+x\d+\.\d+x\d+\.\d+$/', $line)) {
                $currentShipment['no_parcels'] = $matches[1];
                \Log::info("Found parcel count '{$matches[1]}' in line: '$line' for shipment {$currentShipment['parcel_number']}");
            }
            
            // Alternative parcel count extraction (look for patterns like "x1," or "x1 " at start of line)
            if ($currentShipment && preg_match('/^x(\d+)/', $line, $matches)) {
                $currentShipment['no_parcels'] = $matches[1];
                \Log::info("Found start-of-line parcel count '{$matches[1]}' in line: '$line' for shipment {$currentShipment['parcel_number']}");
            }
            
            // Debug: Log lines that might contain parcel information
            if ($currentShipment && (strpos($line, 'x') !== false || strpos($line, 'kg') !== false)) {
                \Log::info("Potential parcel/weight line for shipment {$currentShipment['parcel_number']}: '$line'");
            }
            
            // Extract total amount (look for £ amount at end of line)
            if ($currentShipment && preg_match('/£\s*(\d+\.?\d*)\s*$/', $line, $matches)) {
                $currentShipment['total'] = $matches[1];
            }
            
            // Extract VAT amount
            if ($currentShipment && preg_match('/VAT\s*:\s*£\s*(\d+\.?\d*)/', $line, $matches)) {
                $currentShipment['vat_amount'] = $matches[1];
            }
        }
        
        // Add the last shipment
        if ($currentShipment) {
            $shipments[] = $currentShipment;
        }
        
        return $shipments;
    }
    
    private function extractDepartmentAndAddresses(&$shipment, $lines, $startIndex, $departmentMap)
    {
        $senderAddress = [];
        $receiverAddress = [];
        $foundSender = false;
        $foundReceiver = false;
        
        // Look ahead up to 50 lines for all information
        for ($i = $startIndex + 1; $i < min($startIndex + 50, count($lines)); $i++) {
            $line = trim($lines[$i]);
            
            // Skip empty lines
            if (empty($line)) continue;
            
            // Look for the next shipment to stop processing
            if (preg_match('/^(DHL|TNT|FedEx).*?(\d+)/', $line)) {
                break;
            }
            
            // Look for department in reference lines (more comprehensive)
            if (strpos($line, 'Reference') !== false) {
                foreach ($departmentMap as $code => $mappedCode) {
                    if (stripos($line, $code) !== false) {
                        $shipment['department'] = $mappedCode;
                        break;
                    }
                }
                continue;
            }
            
            // Look for pieces/weight line to identify end of address section
            if (preg_match('/x\d+.*kg/', $line)) {
                break;
            }
            
            // Look for GB / CN, GB / RO patterns (origin/destination) - skip these
            if (preg_match('/GB\s*\/\s*([A-Z]{2})/', $line, $matches)) {
                continue;
            }
            
            // Look for customer number lines - skip these
            if (preg_match('/^\d{8,}/', $line)) {
                continue;
            }
            
            // Look for date patterns - skip these
            if (preg_match('/^\d{2}\/\d{2}\/\d{4}/', $line)) {
                continue;
            }
            
            // Look for ROBERT TODD SONS to identify sender section
            if (stripos($line, 'ROBERT TODD SONS') !== false) {
                $foundSender = true;
                $shipment['sent_from_name'] = 'ROBERT TODD SONS';
                $senderAddress[] = $line;
                continue;
            }
            
            // Look for NARROW LANE, WYMESWOLD, LOUGHBOROUGH to continue sender address
            if ($foundSender && (stripos($line, 'NARROW LANE') !== false || stripos($line, 'WYMESWOLD') !== false || stripos($line, 'LOUGHBOROUGH') !== false || stripos($line, 'LE12 6SD') !== false)) {
                $senderAddress[] = $line;
                continue;
            }
            
            // Look for country patterns to identify receiver address
            if (preg_match('/(United Kingdom|China|Romania|Bulgaria|Turkey|Hong Kong|India|Bangladesh)/', $line, $matches)) {
                if (!$foundReceiver) {
                    $receiverAddress[] = $line;
                    $foundReceiver = true;
                    $shipment['country'] = $matches[1];
                }
                continue;
            }
            
            // Look for receiver name (first line that's not sender-related and not a country)
            if (!$foundReceiver && !preg_match('/^(DHL|TNT|FedEx|Reference|Pieces|Weight|Charges|Customer|Invoice|GB\/|ROBERT|NARROW|WYMESWOLD|LOUGHBOROUGH|LE12|United Kingdom|China|Romania|Bulgaria|Turkey|Hong Kong|India|Bangladesh)/', $line)) {
                if (empty($shipment['destination_name'])) {
                    $shipment['destination_name'] = $line;
                } else {
                    $receiverAddress[] = $line;
                }
            }
            // Continue building receiver address (but only if we've found the receiver name)
            elseif ($foundReceiver && !preg_match('/^(DHL|TNT|FedEx|Reference|Pieces|Weight|Charges|Customer|Invoice|GB\/|ROBERT|NARROW|WYMESWOLD|LOUGHBOROUGH|LE12|x\d+|kg|£|VAT|Package|Fuel|Brexit|Enhanced|Security|Supplement|Economy|Express|G|◀|▶)/', $line)) {
                // Stop collecting address if we hit weight or package information
                if (preg_match('/(\d+\.?\d*)kg|Package|£|x\d+\.?\d*x\d+\.?\d*x\d+\.?\d*|Economy|Express|G|◀|▶/', $line)) {
                    break;
                }
                $receiverAddress[] = $line;
            }
        }
        
        $shipment['sent_from_address'] = implode(', ', array_filter($senderAddress));
        $shipment['destination_address'] = implode(', ', array_filter($receiverAddress));
    }

    public function render()
    {
        // Debug: Log that component is rendering
        \Log::info('ProcessInxpressInvoice component rendering');
        
        return view('livewire.process-inxpress-invoice');
    }
}
