<?php

namespace App\Services;

use App\Models\Price;
use App\Models\PriceResolution;
use App\Models\Colourways;
use App\Models\StyleVersions;
use App\Models\Customers;
use App\Models\Suppliers;
use App\Models\Styles;
use App\Models\Designs;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PriceResolutionService
{
    /**
     * Get or create a price resolution for the given parameters.
     */
    public function resolve(
        int $styleVersionId,
        int $colourwayId,
        ?int $sizeId = null,
        ?int $phaseId = null,
        ?int $seasonId = null
    ): ?PriceResolution {
        // First, try to get an existing fresh resolution
        $resolution = PriceResolution::where([
            'style_versions_id' => $styleVersionId,
            'colourways_id' => $colourwayId,
            'sizes_id' => $sizeId,
            'phase_id' => $phaseId,
            'season_id' => $seasonId,
        ])->fresh()->first();

        if ($resolution) {
            return $resolution;
        }

        // If no fresh resolution exists, compute a new one
        return $this->computeResolution(
            $styleVersionId,
            $colourwayId,
            $sizeId,
            $phaseId,
            $seasonId
        );
    }

    /**
     * Compute a new price resolution and store it in the database.
     */
    public function computeResolution(
        int $styleVersionId,
        int $colourwayId,
        ?int $sizeId = null,
        ?int $phaseId = null,
        ?int $seasonId = null
    ): ?PriceResolution {
        try {
            // Get the colourway to access the price calculation logic
            $colourway = Colourways::with([
                'style_versions.styles.customers',
                'style_versions.factories',
                'style_versions.styles.seasons'
            ])->find($colourwayId);

            if (!$colourway) {
                Log::warning("Colourway not found for resolution: {$colourwayId}");
                return null;
            }

            // Use the existing price calculation logic from Colourways model
            $priceInfo = $colourway->price($sizeId, $phaseId);

            if ($priceInfo['status'] !== 'success' || !$priceInfo['price']) {
                Log::info("No price found for resolution", [
                    'style_version_id' => $styleVersionId,
                    'colourway_id' => $colourwayId,
                    'size_id' => $sizeId,
                    'phase_id' => $phaseId,
                    'status' => $priceInfo['status']
                ]);
                return null;
            }

            $price = $priceInfo['price'];
            
            // Ensure all necessary relationships are loaded for the price calculations
            $price->load([
                'style_versions.styles.customers',
                'style_versions.factories',
                'style_versions.styles.seasons'
            ]);
            
            $styleVersion = $colourway->style_versions;
            $style = $styleVersion->styles;
            $customer = $style->customers;
            $factory = $styleVersion->factories;
            $season = $style->seasons;

            // Prepare resolution data - use try-catch for computed accessors
            $resolutionData = [
                'style_versions_id' => $styleVersionId,
                'colourways_id' => $colourwayId,
                'sizes_id' => $sizeId,
                'phase_id' => $phaseId,
                'season_id' => $seasonId ?? $season->id,
                'customer_id' => $customer->id,
                'supplier_id' => $factory->id,
                'style_id' => $style->id,
                'design_id' => $style->designs_id,
                'price_id' => $price->id,
                'cmt' => $price->cmt,
                'quote' => $price->quote,
                'cmt_status' => $price->cmt_status,
                'quote_status' => $price->quote_status,
                'cust_currency' => $customer->currency,
                'fact_currency' => $factory->currency,
                'fresh_at' => now(),
            ];

            // Safely get computed values
            try {
                $resolutionData['cmt_base'] = $price->cmt_base;
            } catch (\Exception $e) {
                $resolutionData['cmt_base'] = null;
            }

            try {
                $resolutionData['quote_base'] = $price->quote_base;
            } catch (\Exception $e) {
                $resolutionData['quote_base'] = null;
            }

            try {
                $resolutionData['discount_price'] = $price->discount_price;
            } catch (\Exception $e) {
                $resolutionData['discount_price'] = null;
            }

            try {
                $resolutionData['discount_price_base'] = $price->discount_price_base;
            } catch (\Exception $e) {
                $resolutionData['discount_price_base'] = null;
            }

            try {
                $resolutionData['gmt_trans_base'] = $price->gmt_trans_base;
            } catch (\Exception $e) {
                $resolutionData['gmt_trans_base'] = null;
            }

            try {
                $resolutionData['subtotal'] = $price->subtotal;
            } catch (\Exception $e) {
                $resolutionData['subtotal'] = null;
            }

            try {
                $resolutionData['subtotal_base'] = $price->subtotal_base;
            } catch (\Exception $e) {
                $resolutionData['subtotal_base'] = null;
            }

            try {
                $resolutionData['yarn_value_euro'] = $price->yarn_value_euro;
            } catch (\Exception $e) {
                $resolutionData['yarn_value_euro'] = null;
            }

            try {
                $resolutionData['margin'] = $price->margin;
            } catch (\Exception $e) {
                $resolutionData['margin'] = null;
            }

            Log::debug('Creating price resolution', [
                'style_version_id' => $styleVersionId,
                'colourway_id' => $colourwayId,
                'size_id' => $sizeId,
                'phase_id' => $phaseId,
                'price_id' => $price->id,
                'resolution_data' => $resolutionData
            ]);

            // Use updateOrCreate to handle potential race conditions
            $resolution = PriceResolution::updateOrCreate(
                [
                    'style_versions_id' => $styleVersionId,
                    'colourways_id' => $colourwayId,
                    'sizes_id' => $sizeId,
                    'phase_id' => $phaseId,
                    'season_id' => $seasonId ?? $season->id,
                ],
                $resolutionData
            );

            Log::debug("Price resolution computed and stored", [
                'resolution_id' => $resolution->id,
                'style_version_id' => $styleVersionId,
                'colourway_id' => $colourwayId,
                'size_id' => $sizeId,
                'phase_id' => $phaseId
            ]);

            return $resolution;

        } catch (\Exception $e) {
            Log::error("Failed to compute price resolution", [
                'style_version_id' => $styleVersionId,
                'colourway_id' => $colourwayId,
                'size_id' => $sizeId,
                'phase_id' => $phaseId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * Invalidate resolutions based on the given criteria.
     */
    public function invalidateByColourway(int $colourwayId): int
    {
        return PriceResolution::where('colourways_id', $colourwayId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for colourway changes.
     */
    public function invalidateAndWarmupByColourway(int $colourwayId): int
    {
        // Invalidate existing resolutions
        $invalidated = $this->invalidateByColourway($colourwayId);

        // Warm up cache for this colourway
        $this->warmupColourway($colourwayId);

        return $invalidated;
    }

    /**
     * Invalidate resolutions based on customer changes.
     */
    public function invalidateByCustomer(int $customerId): int
    {
        return PriceResolution::where('customer_id', $customerId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for customer changes.
     */
    public function invalidateAndWarmupByCustomer(int $customerId): int
    {
        // Invalidate existing resolutions
        $invalidated = $this->invalidateByCustomer($customerId);

        // Get all colourways for this customer and warm up their cache
        $colourways = Colourways::whereHas('style_versions.styles', function($query) use ($customerId) {
            $query->where('customers_id', $customerId);
        })->pluck('id');

        foreach ($colourways as $colourwayId) {
            $this->warmupColourway($colourwayId);
        }

        return $invalidated;
    }

    /**
     * Invalidate resolutions based on supplier (factory) changes.
     */
    public function invalidateBySupplier(int $supplierId): int
    {
        return PriceResolution::where('supplier_id', $supplierId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for supplier changes.
     */
    public function invalidateAndWarmupBySupplier(int $supplierId): int
    {
        // Invalidate existing resolutions
        $invalidated = $this->invalidateBySupplier($supplierId);

        // Get all colourways for this supplier and warm up their cache
        $colourways = Colourways::whereHas('style_versions', function($query) use ($supplierId) {
            $query->where('factory_id', $supplierId);
        })->pluck('id');

        foreach ($colourways as $colourwayId) {
            $this->warmupColourway($colourwayId);
        }

        return $invalidated;
    }

    /**
     * Invalidate resolutions based on style changes.
     */
    public function invalidateByStyle(int $styleId): int
    {
        return PriceResolution::where('style_id', $styleId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate resolutions based on design changes.
     */
    public function invalidateByDesign(int $designId): int
    {
        return PriceResolution::where('design_id', $designId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for design changes.
     */
    public function invalidateAndWarmupByDesign(int $designId): int
    {
        // Invalidate existing resolutions
        $invalidated = $this->invalidateByDesign($designId);

        // Get all colourways for this design and warm up their cache
        $colourways = Colourways::whereHas('style_versions.styles', function($query) use ($designId) {
            $query->where('designs_id', $designId);
        })->pluck('id');

        foreach ($colourways as $colourwayId) {
            $this->warmupColourway($colourwayId);
        }

        return $invalidated;
    }

    /**
     * Invalidate resolutions based on price changes.
     */
    public function invalidateByPrice(int $priceId): int
    {
        return PriceResolution::where('price_id', $priceId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for price changes.
     */
    public function invalidateAndWarmupByPrice(int $priceId): int
    {
        // Get the price to determine what needs to be warmed up
        $price = Price::find($priceId);
        if (!$price) {
            return 0;
        }

        // Invalidate existing resolutions
        $invalidated = $this->invalidateByPrice($priceId);

        // Invalidate and warm up total cache for affected entities
        $totalService = app(TotalCacheService::class);
        $totalService->invalidateAndWarmupByPrice($price);

        // Automatically warm up cache
        if ($price->colourways_id === null) {
            // For general prices (NULL colourway), warm up cache for all colourways of this style version
            $colourways = Colourways::where('style_versions_id', $price->style_versions_id)->pluck('id');
            foreach ($colourways as $colourwayId) {
                $this->warmupColourway($colourwayId);
            }
        } else {
            // For specific colourway prices, warm up cache for that colourway
            $this->warmupColourway($price->colourways_id);
        }

        return $invalidated;
    }

    /**
     * Invalidate resolutions based on style version changes.
     */
    public function invalidateByStyleVersion(int $styleVersionId): int
    {
        return PriceResolution::where('style_versions_id', $styleVersionId)
            ->update(['fresh_at' => null]);
    }

    /**
     * Invalidate and automatically warm up cache for style version changes.
     */
    public function invalidateAndWarmupByStyleVersion(int $styleVersionId): int
    {
        // Invalidate existing resolutions
        $invalidated = $this->invalidateByStyleVersion($styleVersionId);

        // Warm up cache for all colourways of this style version
        $colourways = Colourways::where('style_versions_id', $styleVersionId)->pluck('id');
        foreach ($colourways as $colourwayId) {
            $this->warmupColourway($colourwayId);
        }

        return $invalidated;
    }

    /**
     * Clean up stale resolutions (older than 7 days).
     */
    public function cleanupStaleResolutions(int $daysOld = 7): int
    {
        $cutoffDate = now()->subDays($daysOld);
        
        return PriceResolution::where('fresh_at', '<', $cutoffDate)
            ->delete();
    }

    /**
     * Invalidate and warm up total cache for entities affected by price changes.
     */
    private function invalidateAndWarmupTotalCacheByPrice(Price $price): void
    {
        try {
            $totalService = app(TotalCacheService::class);
            
            // Get all customer order lines that use this price
            $orderLines = CustomerOrderLines::whereHas('colourways', function($query) use ($price) {
                if ($price->colourways_id === null) {
                    // General price - affects all colourways of this style version
                    $query->where('style_versions_id', $price->style_versions_id);
                } else {
                    // Specific colourway price
                    $query->where('id', $price->colourways_id);
                }
            })->get();

            // Invalidate and warm up total cache for affected order lines
            foreach ($orderLines as $orderLine) {
                $totalService->invalidateEntity('customer_order_line', $orderLine->id);
                $totalService->warmupCustomerOrderLine($orderLine);
                
                // Also invalidate and warm up the parent customer order
                $totalService->invalidateEntity('customer_order', $orderLine->customer_orders_id);
                $totalService->warmupCustomerOrder($orderLine->customer_orders);
            }

            // Get all shipment lines that use this price
            $shipmentLines = ShipmentLine::whereHas('customer_order_lines.colourways', function($query) use ($price) {
                if ($price->colourways_id === null) {
                    // General price - affects all colourways of this style version
                    $query->where('style_versions_id', $price->style_versions_id);
                } else {
                    // Specific colourway price
                    $query->where('id', $price->colourways_id);
                }
            })->get();

            // Invalidate and warm up total cache for affected shipment lines
            foreach ($shipmentLines as $shipmentLine) {
                $totalService->invalidateEntity('shipment_line', $shipmentLine->id);
                $totalService->warmupShipmentLine($shipmentLine);
                
                // Also invalidate and warm up the parent shipment
                if ($shipmentLine->shipment_id) {
                    $totalService->invalidateEntity('shipment', $shipmentLine->shipment_id);
                    $totalService->warmupShipment($shipmentLine->shipment);
                }
            }

        } catch (\Exception $e) {
            Log::error('Failed to invalidate and warm up total cache for price', [
                'price_id' => $price->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Invalidate and warm up total cache for entities affected by colourway changes.
     */
    private function invalidateAndWarmupTotalCacheByColourway(int $colourwayId): void
    {
        try {
            $totalService = app(TotalCacheService::class);
            
            // Get all customer order lines for this colourway
            $orderLines = CustomerOrderLines::where('colourways_id', $colourwayId)->get();

            foreach ($orderLines as $orderLine) {
                $totalService->invalidateEntity('customer_order_line', $orderLine->id);
                $totalService->warmupCustomerOrderLine($orderLine);
                
                // Also invalidate and warm up the parent customer order
                $totalService->invalidateEntity('customer_order', $orderLine->customer_orders_id);
                $totalService->warmupCustomerOrder($orderLine->customer_orders);
            }

            // Get all shipment lines for this colourway
            $shipmentLines = ShipmentLine::whereHas('customer_order_lines', function($query) use ($colourwayId) {
                $query->where('colourways_id', $colourwayId);
            })->get();

            foreach ($shipmentLines as $shipmentLine) {
                $totalService->invalidateEntity('shipment_line', $shipmentLine->id);
                $totalService->warmupShipmentLine($shipmentLine);
                
                // Also invalidate and warm up the parent shipment
                if ($shipmentLine->shipment_id) {
                    $totalService->invalidateEntity('shipment', $shipmentLine->shipment_id);
                    $totalService->warmupShipment($shipmentLine->shipment);
                }
            }

        } catch (\Exception $e) {
            Log::error('Failed to invalidate and warm up total cache for colourway', [
                'colourway_id' => $colourwayId,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get statistics about the resolution cache.
     */
    public function getCacheStats(): array
    {
        $total = PriceResolution::count();
        $fresh = PriceResolution::where('fresh_at', '>', now()->subHours(24))->count();
        $stale = PriceResolution::where(function ($q) {
            $q->whereNull('fresh_at')
              ->orWhere('fresh_at', '<=', now()->subHours(24));
        })->count();

        return [
            'total' => $total,
            'fresh' => $fresh,
            'stale' => $stale,
            'fresh_percentage' => $total > 0 ? round(($fresh / $total) * 100, 2) : 0,
        ];
    }

    /**
     * Warm up the cache for a specific colourway and only relevant sizes/phases.
     */
    public function warmupColourway(int $colourwayId): int
    {
        $colourway = Colourways::with(['style_versions.styles.seasons', 'style_versions.factories', 'style_versions.styles.customers'])->find($colourwayId);
        
        if (!$colourway) {
            return 0;
        }

        $seasonId = $colourway->style_versions->styles->seasons_id ?? null;
        $styleVersionId = $colourway->style_versions_id;
        $computed = 0;

        // Skip colourways without seasons
        if (!$seasonId) {
            \Log::warning("Skipping colourway {$colourwayId} - no season found");
            return 0;
        }

        // Get all sizes that are used in customer order line quantities for this colourway
        $sizesUsed = DB::table('customer_order_line_quantities as colq')
            ->join('customer_order_lines as col', 'colq.customer_order_lines_id', '=', 'col.id')
            ->where('col.colourways_id', $colourwayId)
            ->distinct()
            ->pluck('colq.sizes_id');

        // Get all phases that are used in prices for this style version
        $phasesUsed = DB::table('prices')
            ->where('style_versions_id', $styleVersionId)
            ->where(function ($query) use ($colourwayId) {
                $query->where('colourways_id', $colourwayId)
                      ->orWhereNull('colourways_id');
            })
            ->whereNotNull('phase_id')
            ->distinct()
            ->pluck('phase_id');

        // Always include null size and phase for generic prices
        $sizesToProcess = $sizesUsed->toArray();
        $sizesToProcess[] = null;
        
        $phasesToProcess = $phasesUsed->toArray();
        $phasesToProcess[] = null;

        // If no sizes are used, still process generic prices (null size)
        if (empty($sizesUsed)) {
            $sizesToProcess = [null];
        }

        // Compute resolution for each size/phase combination using the actual price() method
        foreach ($sizesToProcess as $sizeId) {
            foreach ($phasesToProcess as $phaseId) {
                try {
                    // Use the actual Colourways price() method to get the price data
                    $priceData = $colourway->price($sizeId, $phaseId);
                    
                    if ($priceData && isset($priceData['price']) && $priceData['price']) {
                        $price = $priceData['price'];
                        
                        // Create resolution using the actual price data
                        $resolution = $this->createResolutionFromPrice($colourway, $price, $sizeId, $phaseId, $seasonId);
                        if ($resolution) {
                            $computed++;
                        }
                    }
                } catch (\Exception $e) {
                    \Log::warning("Failed to get price for colourway {$colourwayId}, size {$sizeId}, phase {$phaseId}: " . $e->getMessage());
                }
            }
        }

        return $computed;
    }

    /**
     * Create a price resolution from the actual price data returned by Colourways::price()
     */
    private function createResolutionFromPrice(Colourways $colourway, $price, ?int $sizeId, ?int $phaseId, ?int $seasonId): ?PriceResolution
    {
        try {
            $styleVersion = $colourway->style_versions;
            $style = $styleVersion->styles;
            $customer = $style->customers;
            $factory = $styleVersion->factories;

            // Check if resolution already exists
            $existing = PriceResolution::where('style_versions_id', $styleVersion->id)
                ->where('colourways_id', $colourway->id)
                ->where('sizes_id', $sizeId)
                ->where('phase_id', $phaseId)
                ->where('season_id', $seasonId)
                ->whereNotNull('fresh_at')
                ->first();

            if ($existing) {
                return $existing;
            }

            // Create new resolution
            $resolution = PriceResolution::create([
                'style_versions_id' => $styleVersion->id,
                'colourways_id' => $colourway->id,
                'sizes_id' => $sizeId,
                'phase_id' => $phaseId,
                'season_id' => $seasonId,
                'customer_id' => $customer->id,
                'supplier_id' => $factory->id,
                'style_id' => $style->id,
                'design_id' => $style->designs_id,
                'price_id' => $price->id,
                'cmt' => $price->cmt,
                'quote' => $price->quote,
                'cmt_status' => $price->cmt_status,
                'quote_status' => $price->quote_status,
                'cust_currency' => $customer->currency,
                'fact_currency' => $factory->currency,
                'fresh_at' => now(),
            ]);

            // Safely get computed values
            try {
                $resolution->cmt_base = $price->cmt_base;
            } catch (\Exception $e) {
                $resolution->cmt_base = null;
            }

            try {
                $resolution->quote_base = $price->quote_base;
            } catch (\Exception $e) {
                $resolution->quote_base = null;
            }

            try {
                $resolution->discount_price = $price->discount_price;
            } catch (\Exception $e) {
                $resolution->discount_price = null;
            }

            try {
                $resolution->discount_price_base = $price->discount_price_base;
            } catch (\Exception $e) {
                $resolution->discount_price_base = null;
            }

            try {
                $resolution->subtotal = $price->subtotal;
            } catch (\Exception $e) {
                $resolution->subtotal = null;
            }

            try {
                $resolution->subtotal_base = $price->subtotal_base;
            } catch (\Exception $e) {
                $resolution->subtotal_base = null;
            }

            try {
                $resolution->gmt_trans_base = $price->gmt_trans_base;
            } catch (\Exception $e) {
                $resolution->gmt_trans_base = null;
            }

            try {
                $resolution->yarn_value_euro = $price->yarn_value_euro;
            } catch (\Exception $e) {
                $resolution->yarn_value_euro = null;
            }

            try {
                $resolution->margin = $price->margin;
            } catch (\Exception $e) {
                $resolution->margin = null;
            }

            $resolution->save();

            return $resolution;

        } catch (\Exception $e) {
            \Log::error("Failed to create resolution for colourway {$colourway->id}: " . $e->getMessage());
            return null;
        }
    }
}
