const { Pool } = require('pg');
const bcrypt = require('bcrypt');

// Database configuration
const dbConfig = process.env.INSTANCE_UNIX_SOCKET
  ? {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.INSTANCE_UNIX_SOCKET,
    }
  : {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10),
    };

const pool = new Pool(dbConfig);

// Subscription tier limits
const TIER_LIMITS = {
  free: {
    maxPhotos: 50,
    maxStorageBytes: 500 * 1024 * 1024, // 500MB
    maxAlbums: 1,
    price: 0, // £0.00
    customDomain: false
  },
  starter: {
    maxPhotos: 1000,
    maxStorageBytes: 5 * 1024 * 1024 * 1024, // 5GB
    maxAlbums: null, // unlimited
    price: 1000, // £10.00 in pence
    customDomain: false
  },
  professional: {
    maxPhotos: 10000,
    maxStorageBytes: 50 * 1024 * 1024 * 1024, // 50GB
    maxAlbums: null, // unlimited
    price: 2500, // £25.00 in pence
    customDomain: true
  },
  business: {
    maxPhotos: 30000,
    maxStorageBytes: 150 * 1024 * 1024 * 1024, // 150GB
    maxAlbums: null, // unlimited
    price: 7500, // £75.00 in pence
    customDomain: true
  }
};

// ============================================================================
// USER MANAGEMENT
// ============================================================================

/**
 * Create a new user account
 */
async function createUser(username, email, password, fullName = null) {
  const passwordHash = await bcrypt.hash(password, 10);
  const trialEndsAt = new Date(Date.now() + 14 * 24 * 60 * 60 * 1000); // 14 days from now
  
  const result = await pool.query(
    `INSERT INTO users (username, email, password_hash, full_name, subscription_tier, subscription_status, trial_ends_at)
     VALUES ($1, $2, $3, $4, 'starter', 'trialing', $5)
     RETURNING id, username, email, full_name, subscription_tier, subscription_status, trial_ends_at, created_at`,
    [username.toLowerCase().trim(), email.toLowerCase().trim(), passwordHash, fullName, trialEndsAt]
  );
  
  return result.rows[0];
}

/**
 * Get user by ID
 */
async function getUserById(userId) {
  const result = await pool.query(
    'SELECT * FROM users WHERE id = $1',
    [userId]
  );
  return result.rows[0] || null;
}

/**
 * Get user by username (for subdomain lookup)
 */
async function getUserByUsername(username) {
  const result = await pool.query(
    'SELECT * FROM users WHERE username = $1',
    [username.toLowerCase().trim()]
  );
  return result.rows[0] || null;
}

/**
 * Get user by custom domain
 */
async function getUserByCustomDomain(domain) {
  const result = await pool.query(
    'SELECT * FROM users WHERE custom_domain = $1 AND custom_domain_verified = TRUE',
    [domain.toLowerCase().trim()]
  );
  return result.rows[0] || null;
}

/**
 * Get user by email
 */
async function getUserByEmail(email) {
  const result = await pool.query(
    'SELECT * FROM users WHERE email = $1',
    [email.toLowerCase().trim()]
  );
  return result.rows[0] || null;
}

/**
 * Verify user password
 */
async function verifyUserPassword(email, password) {
  const user = await getUserByEmail(email);
  if (!user) return null;
  
  const isValid = await bcrypt.compare(password, user.password_hash);
  if (!isValid) return null;
  
  // Update last login
  await pool.query(
    'UPDATE users SET last_login_at = CURRENT_TIMESTAMP WHERE id = $1',
    [user.id]
  );
  
  return user;
}

/**
 * Update user profile
 */
async function updateUserProfile(userId, updates) {
  const allowedFields = ['full_name', 'email', 'site_name', 'primary_color', 'logo_url', 
                         'hot_chocolate_default_url', 'hot_chocolate_text', 'contact_email', 
                         'social_links', 'allow_downloads', 'require_age_verification'];
  
  const fields = [];
  const values = [];
  let paramIndex = 1;
  
  for (const [key, value] of Object.entries(updates)) {
    if (allowedFields.includes(key)) {
      fields.push(`${key} = $${paramIndex}`);
      values.push(key === 'social_links' ? JSON.stringify(value) : value);
      paramIndex++;
    }
  }
  
  if (fields.length === 0) return;
  
  values.push(userId);
  await pool.query(
    `UPDATE users SET ${fields.join(', ')}, updated_at = CURRENT_TIMESTAMP WHERE id = $${paramIndex}`,
    values
  );
}

/**
 * Update user password
 */
async function updateUserPassword(userId, newPassword) {
  const passwordHash = await bcrypt.hash(newPassword, 10);
  await pool.query(
    'UPDATE users SET password_hash = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
    [passwordHash, userId]
  );
}

/**
 * Check if user has reached photo limit
 */
async function hasReachedPhotoLimit(userId) {
  const user = await getUserById(userId);
  if (!user) return true;
  
  // Convert to numbers (PostgreSQL may return as string)
  const current = parseInt(user.current_photo_count || '0', 10);
  const max = parseInt(user.max_photos || '0', 10);
  
  return current >= max;
}

/**
 * Check if user has reached storage limit
 */
async function hasReachedStorageLimit(userId, additionalBytes = 0) {
  const user = await getUserById(userId);
  if (!user) {
    console.log(`[hasReachedStorageLimit] User ${userId} not found!`);
    return true;
  }
  
  // Convert to numbers (PostgreSQL bigint returns as string)
  const current = parseInt(user.current_storage_bytes || '0', 10);
  const max = parseInt(user.max_storage_bytes || '0', 10);
  const additional = parseInt(additionalBytes || '0', 10);
  const total = current + additional;
  const wouldExceed = total > max;
  
  console.log(`[hasReachedStorageLimit] User ${user.username} (${userId})`);
  console.log(`  Current storage: ${current} bytes (${(current / 1024 / 1024 / 1024).toFixed(3)}GB)`);
  console.log(`  Additional: ${additional} bytes (${(additional / 1024 / 1024).toFixed(2)}MB)`);
  console.log(`  Total would be: ${total} bytes (${(total / 1024 / 1024 / 1024).toFixed(3)}GB)`);
  console.log(`  Max allowed: ${max} bytes (${(max / 1024 / 1024 / 1024).toFixed(3)}GB)`);
  console.log(`  Would exceed? ${wouldExceed}`);
  
  return wouldExceed;
}

/**
 * Get user usage statistics
 */
async function getUserUsageStats(userId) {
  const user = await getUserById(userId);
  if (!user) return null;
  
  // Convert to numbers (PostgreSQL bigint returns as string)
  const currentPhotos = parseInt(user.current_photo_count || '0', 10);
  const maxPhotos = parseInt(user.max_photos || '0', 10);
  const currentStorageBytes = parseInt(user.current_storage_bytes || '0', 10);
  const maxStorageBytes = parseInt(user.max_storage_bytes || '0', 10);
  
  return {
    currentPhotos,
    maxPhotos,
    currentStorageBytes,
    maxStorageBytes,
    photoUsagePercent: maxPhotos > 0 ? Math.round((currentPhotos / maxPhotos) * 100) : 0,
    storageUsagePercent: maxStorageBytes > 0 ? Math.round((currentStorageBytes / maxStorageBytes) * 100) : 0
  };
}

/**
 * Update user usage after uploading/deleting photos
 */
async function updateUserUsage(userId, photoCountDelta, storageBytesDelta) {
  await pool.query(
    `UPDATE users 
     SET current_photo_count = current_photo_count + $1,
         current_storage_bytes = current_storage_bytes + $2,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $3`,
    [photoCountDelta, storageBytesDelta, userId]
  );
}

/**
 * Update subscription tier
 */
async function updateSubscriptionTier(userId, tier, stripeSubscriptionId = null) {
  const limits = TIER_LIMITS[tier];
  if (!limits) {
    throw new Error(`Invalid subscription tier: ${tier}`);
  }
  
  await pool.query(
    `UPDATE users 
     SET subscription_tier = $1, 
         subscription_status = 'active',
         stripe_subscription_id = $2,
         max_photos = $3,
         max_storage_bytes = $4,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $5`,
    [tier, stripeSubscriptionId, limits.maxPhotos, limits.maxStorageBytes, userId]
  );
}

/**
 * Update Stripe customer ID
 */
async function updateStripeCustomerId(userId, stripeCustomerId) {
  await pool.query(
    'UPDATE users SET stripe_customer_id = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
    [stripeCustomerId, userId]
  );
}

/**
 * Set custom domain for user
 */
async function setCustomDomain(userId, domain) {
  await pool.query(
    `UPDATE users 
     SET custom_domain = $1, 
         custom_domain_verified = FALSE,
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $2`,
    [domain.toLowerCase().trim(), userId]
  );
}

/**
 * Verify custom domain
 */
async function verifyCustomDomain(userId) {
  await pool.query(
    'UPDATE users SET custom_domain_verified = TRUE, updated_at = CURRENT_TIMESTAMP WHERE id = $1',
    [userId]
  );
}

// ============================================================================
// SESSION MANAGEMENT
// ============================================================================

/**
 * Create a new session
 */
async function createSession(userId, sessionId, expiresAt, userAgent = null, ipAddress = null) {
  const result = await pool.query(
    `INSERT INTO sessions (session_id, user_id, expires_at, user_agent, ip_address)
     VALUES ($1, $2, $3, $4, $5)
     RETURNING *`,
    [sessionId, userId, expiresAt, userAgent, ipAddress]
  );
  return result.rows[0];
}

/**
 * Get session by session ID
 */
async function getSession(sessionId) {
  const result = await pool.query(
    `SELECT s.*, u.* 
     FROM sessions s
     JOIN users u ON s.user_id = u.id
     WHERE s.session_id = $1 AND s.expires_at > CURRENT_TIMESTAMP`,
    [sessionId]
  );
  return result.rows[0] || null;
}

/**
 * Delete session (logout)
 */
async function deleteSession(sessionId) {
  await pool.query('DELETE FROM sessions WHERE session_id = $1', [sessionId]);
}

/**
 * Delete all user sessions
 */
async function deleteAllUserSessions(userId) {
  await pool.query('DELETE FROM sessions WHERE user_id = $1', [userId]);
}

/**
 * Clean up expired sessions
 */
async function cleanupExpiredSessions() {
  await pool.query('DELETE FROM sessions WHERE expires_at < CURRENT_TIMESTAMP');
}

// ============================================================================
// ALBUM MANAGEMENT (Multi-tenant)
// ============================================================================

async function createAlbum(userId, title, code) {
  const result = await pool.query(
    'INSERT INTO albums (user_id, title, code) VALUES ($1, $2, $3) RETURNING id',
    [userId, title, code]
  );
  return result.rows[0].id;
}

async function listAlbums(userId) {
  const result = await pool.query(
    'SELECT id, title, code, created_at FROM albums WHERE user_id = $1 ORDER BY sort_order ASC, created_at DESC',
    [userId]
  );
  return result.rows;
}

async function getAlbumById(albumId, userId = null) {
  const query = userId
    ? 'SELECT * FROM albums WHERE id = $1 AND user_id = $2'
    : 'SELECT * FROM albums WHERE id = $1';
  const params = userId ? [albumId, userId] : [albumId];
  
  const result = await pool.query(query, params);
  return result.rows[0] || null;
}

async function getAlbumByCode(code, userId = null) {
  const query = userId
    ? 'SELECT * FROM albums WHERE code = $1 AND user_id = $2'
    : 'SELECT * FROM albums WHERE code = $1';
  const params = userId ? [code, userId] : [code];
  
  const result = await pool.query(query, params);
  return result.rows[0] || null;
}

async function updateAlbumTitle(albumId, userId, title) {
  await pool.query(
    'UPDATE albums SET title = $1 WHERE id = $2 AND user_id = $3',
    [title, albumId, userId]
  );
}

async function updateAlbumDescription(albumId, userId, description) {
  await pool.query(
    'UPDATE albums SET description = $1 WHERE id = $2 AND user_id = $3',
    [description, albumId, userId]
  );
}

async function toggleAlbumPublic(albumId, userId, isPublic) {
  await pool.query(
    'UPDATE albums SET is_public = $1 WHERE id = $2 AND user_id = $3',
    [isPublic ? 1 : 0, albumId, userId]
  );
}

async function updateHotChocolateSettings(albumId, userId, enabled, url) {
  await pool.query(
    'UPDATE albums SET hot_choc_enabled = $1, hot_choc_url = $2 WHERE id = $3 AND user_id = $4',
    [enabled ? 1 : 0, url || null, albumId, userId]
  );
}

async function setHomePageAlbum(albumId, userId) {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    // Clear any existing home page album for this user
    await client.query('UPDATE albums SET is_home_page_album = 0 WHERE user_id = $1', [userId]);
    // Set the new one
    await client.query(
      'UPDATE albums SET is_home_page_album = 1 WHERE id = $1 AND user_id = $2',
      [albumId, userId]
    );
    await client.query('COMMIT');
  } catch (err) {
    await client.query('ROLLBACK');
    throw err;
  } finally {
    client.release();
  }
}

async function getHomePageAlbum(userId) {
  const result = await pool.query(
    'SELECT * FROM albums WHERE is_home_page_album = 1 AND user_id = $1',
    [userId]
  );
  return result.rows[0] || null;
}

async function updateAlbumSortOrder(albumIds, userId) {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    for (let i = 0; i < albumIds.length; i++) {
      await client.query(
        'UPDATE albums SET sort_order = $1 WHERE id = $2 AND user_id = $3',
        [i, albumIds[i], userId]
      );
    }
    await client.query('COMMIT');
  } catch (err) {
    await client.query('ROLLBACK');
    throw err;
  } finally {
    client.release();
  }
}

async function deleteAlbum(albumId, userId) {
  await pool.query('DELETE FROM albums WHERE id = $1 AND user_id = $2', [albumId, userId]);
}

async function listPublicAlbums(userId) {
  const result = await pool.query(`
    SELECT a.*, 
           (SELECT p.filename 
            FROM photos p 
            WHERE p.album_id = a.id 
            ORDER BY p.sort_order ASC, p.created_at ASC 
            LIMIT 1) as cover_filename,
           (SELECT COUNT(*) FROM photos p WHERE p.album_id = a.id) as photo_count
    FROM albums a 
    WHERE a.is_public = 1 AND a.user_id = $1
    ORDER BY a.sort_order ASC, a.created_at DESC
  `, [userId]);
  return result.rows;
}

// ============================================================================
// PHOTO MANAGEMENT (Multi-tenant)
// ============================================================================

async function addPhoto(albumId, userId, filename, originalName, mimeType, sizeBytes) {
  // Verify album belongs to user
  const album = await getAlbumById(albumId, userId);
  if (!album) throw new Error('Album not found or does not belong to user');
  
  const result = await pool.query(
    'INSERT INTO photos (album_id, filename, original_name, mime_type, size_bytes) VALUES ($1, $2, $3, $4, $5) RETURNING id',
    [albumId, filename, originalName, mimeType, sizeBytes]
  );
  return result.rows[0].id;
}

async function getPhotosByAlbum(albumId) {
  const result = await pool.query(
    'SELECT * FROM photos WHERE album_id = $1 ORDER BY sort_order ASC, created_at DESC',
    [albumId]
  );
  return result.rows;
}

async function getPhotoById(photoId) {
  const result = await pool.query('SELECT * FROM photos WHERE id = $1', [photoId]);
  return result.rows[0] || null;
}

async function updatePhotoSortOrder(photoIds) {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    for (let i = 0; i < photoIds.length; i++) {
      await client.query('UPDATE photos SET sort_order = $1 WHERE id = $2', [i, photoIds[i]]);
    }
    await client.query('COMMIT');
  } catch (err) {
    await client.query('ROLLBACK');
    throw err;
  } finally {
    client.release();
  }
}

async function deletePhoto(photoId, userId) {
  // Verify photo belongs to user via album
  await pool.query(
    `DELETE FROM photos 
     WHERE id = $1 
     AND album_id IN (SELECT id FROM albums WHERE user_id = $2)`,
    [photoId, userId]
  );
}

// ============================================================================
// GROUP MANAGEMENT (Multi-tenant)
// ============================================================================

async function createGroup(userId, name, code, description = null) {
  const result = await pool.query(
    'INSERT INTO album_groups (user_id, name, code, description) VALUES ($1, $2, $3, $4) RETURNING id',
    [userId, name, code, description]
  );
  return result.rows[0].id;
}

async function listGroups(userId) {
  const result = await pool.query(
    'SELECT id, name, code, description, created_at FROM album_groups WHERE user_id = $1 ORDER BY created_at DESC',
    [userId]
  );
  return result.rows;
}

async function getGroupById(groupId, userId = null) {
  const query = userId
    ? 'SELECT * FROM album_groups WHERE id = $1 AND user_id = $2'
    : 'SELECT * FROM album_groups WHERE id = $1';
  const params = userId ? [groupId, userId] : [groupId];
  
  const result = await pool.query(query, params);
  return result.rows[0] || null;
}

async function getGroupByCode(code, userId = null) {
  const query = userId
    ? 'SELECT * FROM album_groups WHERE code = $1 AND user_id = $2'
    : 'SELECT * FROM album_groups WHERE code = $1';
  const params = userId ? [code, userId] : [code];
  
  const result = await pool.query(query, params);
  return result.rows[0] || null;
}

async function getAlbumsByGroup(groupId) {
  const result = await pool.query(
    'SELECT * FROM albums WHERE group_id = $1 ORDER BY sort_order ASC, created_at DESC',
    [groupId]
  );
  return result.rows;
}

async function assignAlbumToGroup(albumId, userId, groupId) {
  await pool.query(
    'UPDATE albums SET group_id = $1 WHERE id = $2 AND user_id = $3',
    [groupId, albumId, userId]
  );
}

async function removeAlbumFromGroup(albumId, userId) {
  await pool.query(
    'UPDATE albums SET group_id = NULL WHERE id = $1 AND user_id = $2',
    [albumId, userId]
  );
}

async function getGroupForAlbum(albumId) {
  const result = await pool.query(
    'SELECT ag.* FROM album_groups ag JOIN albums a ON a.group_id = ag.id WHERE a.id = $1',
    [albumId]
  );
  return result.rows[0] || null;
}

async function deleteGroup(groupId, userId, deleteAlbums = false) {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    
    if (deleteAlbums) {
      await client.query('DELETE FROM albums WHERE group_id = $1 AND user_id = $2', [groupId, userId]);
    } else {
      await client.query('UPDATE albums SET group_id = NULL WHERE group_id = $1 AND user_id = $2', [groupId, userId]);
    }
    
    await client.query('DELETE FROM album_groups WHERE id = $1 AND user_id = $2', [groupId, userId]);
    
    await client.query('COMMIT');
  } catch (err) {
    await client.query('ROLLBACK');
    throw err;
  } finally {
    client.release();
  }
}

// ============================================================================
// LEGACY FUNCTIONS (for backwards compatibility)
// ============================================================================

async function getSiteConfig() {
  // This is deprecated in multi-tenant mode
  // Return empty config - user-specific config is in users table
  return {
    siteName: '',
    primaryColor: '',
    logoUrl: '',
    hotChocolateDefaultUrl: '',
    hotChocolateText: '',
    contactEmail: '',
    socialLinks: {}
  };
}

async function updateSiteConfig(updates) {
  // This is deprecated in multi-tenant mode
  // Use updateUserProfile instead
  console.warn('updateSiteConfig is deprecated in multi-tenant mode. Use updateUserProfile instead.');
}

module.exports = {
  // User management
  createUser,
  getUserById,
  getUserByUsername,
  getUserByCustomDomain,
  getUserByEmail,
  verifyUserPassword,
  updateUserProfile,
  updateUserPassword,
  hasReachedPhotoLimit,
  hasReachedStorageLimit,
  getUserUsageStats,
  updateUserUsage,
  updateSubscriptionTier,
  updateStripeCustomerId,
  setCustomDomain,
  verifyCustomDomain,
  
  // Session management
  createSession,
  getSession,
  deleteSession,
  deleteAllUserSessions,
  cleanupExpiredSessions,
  
  // Album management
  createAlbum,
  listAlbums,
  getAlbumById,
  getAlbumByCode,
  updateAlbumTitle,
  updateAlbumDescription,
  toggleAlbumPublic,
  updateHotChocolateSettings,
  setHomePageAlbum,
  getHomePageAlbum,
  updateAlbumSortOrder,
  deleteAlbum,
  listPublicAlbums,
  
  // Photo management
  addPhoto,
  getPhotosByAlbum,
  getPhotoById,
  updatePhotoSortOrder,
  deletePhoto,
  
  // Group management
  createGroup,
  listGroups,
  getGroupById,
  getGroupByCode,
  getAlbumsByGroup,
  assignAlbumToGroup,
  removeAlbumFromGroup,
  getGroupForAlbum,
  deleteGroup,
  
  // Legacy (backwards compatibility)
  getSiteConfig,
  updateSiteConfig,
  
  // Utility
  pool,
  TIER_LIMITS
};

