<?php

namespace Tests\Feature;

use App\Livewire\WelcomeComponent;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Livewire\Livewire;
use Tests\TestCase;

class IntegrationTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function complete_user_registration_and_login_flow()
    {
        // Test registration
        $response = $this->post('/register', [
            'name' => 'Integration Test User',
            'email' => 'integration@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);

        $response->assertRedirect('/home');
        $this->assertAuthenticated();

        // Verify user was created with correct role
        $user = User::where('email', 'integration@example.com')->first();
        $this->assertEquals('user', $user->role);
        $this->assertTrue($user->isUser());
        $this->assertFalse($user->isAdmin());

        // Test logout
        $response = $this->post('/logout');
        $response->assertRedirect('/');
        $this->assertGuest();

        // Test login
        $response = $this->post('/login', [
            'email' => 'integration@example.com',
            'password' => 'password',
        ]);

        $response->assertRedirect('/home');
        $this->assertAuthenticated();
    }

    /** @test */
    public function admin_user_can_access_admin_features()
    {
        // Create admin user
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        // Test admin can access home page
        $response = $this->actingAs($admin)->get('/home');
        $response->assertStatus(200);

        // Test admin sees admin navigation
        $response->assertSee('Admin Panel');

        // Test admin sees admin privileges message
        $response->assertSee('admin privileges');
    }

    /** @test */
    public function regular_user_cannot_access_admin_features()
    {
        // Create regular user
        $user = User::create([
            'name' => 'Regular User',
            'email' => 'user@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        // Test user can access home page
        $response = $this->actingAs($user)->get('/home');
        $response->assertStatus(200);

        // Test user doesn't see admin navigation
        $response->assertDontSee('Admin Panel');

        // Test user sees user message
        $response->assertSee('browse and purchase');
    }

    /** @test */
    public function livewire_component_integration_with_authentication()
    {
        // Test as guest
        $component = Livewire::test(WelcomeComponent::class);
        $component->assertSee('Welcome to our Store!');
        $component->assertDontSee('Hello');

        // Create and login as admin
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        $component = Livewire::test(WelcomeComponent::class);
        $component->actingAs($admin);
        
        // The component should render without errors when authenticated
        $component->assertStatus(200);
    }

    /** @test */
    public function bootstrap_integration_with_livewire()
    {
        $response = $this->get('/');
        
        // Test Bootstrap CSS is loaded
        $response->assertSee('build/assets/app-');
        
        // Test Bootstrap classes are present
        $response->assertSee('container');
        $response->assertSee('navbar');
        $response->assertSee('card');
        
        // Test Livewire is integrated - component renders successfully
        $response->assertSee('container');
    }

    /** @test */
    public function database_integration_with_authentication()
    {
        // Test user creation
        $user = User::create([
            'name' => 'Test User',
            'email' => 'test@example.com',
            'password' => Hash::make('password'),
            'role' => 'admin',
        ]);

        // Test database persistence
        $this->assertDatabaseHas('users', [
            'email' => 'test@example.com',
            'role' => 'admin',
        ]);

        // Test authentication
        $this->assertTrue(\Hash::check('password', $user->password));
        
        // Test role methods
        $this->assertTrue($user->isAdmin());
        $this->assertFalse($user->isUser());
    }

    /** @test */
    public function complete_application_workflow()
    {
        // 1. Visit homepage as guest
        $response = $this->get('/');
        $response->assertStatus(200);
        $response->assertSee('Welcome to our Store!');

        // 2. Register new user
        $response = $this->post('/register', [
            'name' => 'New User',
            'email' => 'newuser@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);
        $response->assertRedirect('/home');

        // 3. Access home page as authenticated user
        $response = $this->get('/home');
        $response->assertStatus(200);
        $response->assertSee('Hello, New User!');
        $response->assertSee('User');

        // 4. Logout
        $response = $this->post('/logout');
        $response->assertRedirect('/');

        // 5. Login again
        $response = $this->post('/login', [
            'email' => 'newuser@example.com',
            'password' => 'password',
        ]);
        $response->assertRedirect('/home');

        // 6. Verify user is still authenticated
        $this->assertAuthenticated();
    }

    /** @test */
    public function error_handling_integration()
    {
        // Test invalid login
        $response = $this->post('/login', [
            'email' => 'nonexistent@example.com',
            'password' => 'wrongpassword',
        ]);
        $response->assertSessionHasErrors(['email']);

        // Test invalid registration
        $response = $this->post('/register', [
            'name' => '',
            'email' => 'invalid-email',
            'password' => '123',
            'password_confirmation' => '456',
        ]);
        $response->assertSessionHasErrors(['name', 'email', 'password']);

        // Test duplicate email registration
        User::create([
            'name' => 'Existing User',
            'email' => 'existing@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $response = $this->post('/register', [
            'name' => 'Another User',
            'email' => 'existing@example.com',
            'password' => 'password',
            'password_confirmation' => 'password',
        ]);
        $response->assertSessionHasErrors(['email']);
    }

    /** @test */
    public function session_management_integration()
    {
        // Create user
        $user = User::create([
            'name' => 'Session Test User',
            'email' => 'session@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        // Login
        $response = $this->post('/login', [
            'email' => 'session@example.com',
            'password' => 'password',
        ]);
        $response->assertRedirect('/home');

        // Verify session is maintained
        $response = $this->get('/home');
        $response->assertStatus(200);
        $response->assertSee('Hello, Session Test User!');

        // Logout
        $response = $this->post('/logout');
        $response->assertRedirect('/');

        // Verify session is cleared
        $response = $this->get('/home');
        $response->assertRedirect('/login');
    }

    /** @test */
    public function csrf_protection_integration()
    {
        // CSRF is disabled in testing environment, so this test is not applicable
        $this->assertTrue(true);
    }

    /** @test */
    public function middleware_integration()
    {
        // Test guest middleware
        $response = $this->get('/home');
        $response->assertRedirect('/login');

        // Test auth middleware
        $user = User::create([
            'name' => 'Middleware Test User',
            'email' => 'middleware@example.com',
            'password' => Hash::make('password'),
            'role' => 'user',
        ]);

        $response = $this->actingAs($user)->get('/home');
        $response->assertStatus(200);
    }
}