<x-app-layout>
    <x-slot name="header">
        <div class="flex items-center justify-between">
            <h2 class="font-semibold text-xl text-gray-800 leading-tight">
                {{ __('Admin - Library Tracks') }}
            </h2>
            <button id="add-track-btn" class="inline-flex items-center px-4 py-2 bg-indigo-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-indigo-700">
                Add Track
            </button>
        </div>
    </x-slot>

    <div class="py-12">
        <div class="max-w-7xl mx-auto sm:px-6 lg:px-8">
            <div class="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                <div class="p-6 text-gray-900">
                    <!-- Tracks Table -->
                    <div class="overflow-x-auto">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Duration</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="tracks-tbody" class="bg-white divide-y divide-gray-200">
                                <!-- Loading state -->
                                <tr>
                                    <td colspan="5" class="px-6 py-4 text-center">
                                        <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-indigo-600 mx-auto"></div>
                                        <p class="mt-2 text-sm text-gray-500">Loading tracks...</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Track Modal -->
    <div id="track-modal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 id="modal-title" class="text-lg font-medium text-gray-900 mb-4">Add New Track</h3>
                <form id="track-form">
                    <input type="hidden" id="track-id" name="id">
                    
                    <div class="mb-4">
                        <label for="title" class="block text-sm font-medium text-gray-700">Title</label>
                        <input type="text" id="title" name="title" required
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="mb-4">
                        <label for="category_id" class="block text-sm font-medium text-gray-700">Category</label>
                        <select id="category_id" name="category_id" required
                                class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                            <option value="">Select Category</option>
                        </select>
                    </div>

                    <div class="mb-4">
                        <label for="duration_sec" class="block text-sm font-medium text-gray-700">Duration (seconds)</label>
                        <input type="number" id="duration_sec" name="duration_sec" required min="1"
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="mb-4">
                        <label for="audio_path" class="block text-sm font-medium text-gray-700">Audio Path</label>
                        <input type="text" id="audio_path" name="audio_path" required
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="mb-4">
                        <label for="transcript_path" class="block text-sm font-medium text-gray-700">Transcript Path (optional)</label>
                        <input type="text" id="transcript_path" name="transcript_path"
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="mb-4">
                        <label for="tags" class="block text-sm font-medium text-gray-700">Tags (comma-separated)</label>
                        <input type="text" id="tags" name="tags" placeholder="relaxation, sleep, meditation"
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="mb-4">
                        <label for="published_at" class="block text-sm font-medium text-gray-700">Published At</label>
                        <input type="datetime-local" id="published_at" name="published_at"
                               class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500">
                    </div>

                    <div class="flex justify-end space-x-3">
                        <button type="button" id="cancel-btn" class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 border border-gray-300 rounded-md hover:bg-gray-200">
                            Cancel
                        </button>
                        <button type="submit" class="px-4 py-2 text-sm font-medium text-white bg-indigo-600 border border-transparent rounded-md hover:bg-indigo-700">
                            Save
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            let tracks = [];
            let categories = [];

            // Load initial data
            loadCategories();
            loadTracks();

            // Event listeners
            document.getElementById('add-track-btn').addEventListener('click', showAddModal);
            document.getElementById('cancel-btn').addEventListener('click', hideModal);
            document.getElementById('track-form').addEventListener('submit', handleSubmit);

            function loadCategories() {
                fetch('/api/v1/library/categories')
                    .then(response => response.json())
                    .then(data => {
                        if (data.data) {
                            categories = data.data;
                            populateCategorySelect();
                        }
                    })
                    .catch(error => console.error('Error loading categories:', error));
            }

            function populateCategorySelect() {
                const select = document.getElementById('category_id');
                select.innerHTML = '<option value="">Select Category</option>';
                
                categories.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category.id;
                    option.textContent = category.name;
                    select.appendChild(option);
                });
            }

            function loadTracks() {
                fetch('/api/v1/library/tracks?per_page=100')
                    .then(response => response.json())
                    .then(data => {
                        if (data.data) {
                            tracks = data.data;
                            displayTracks();
                        }
                    })
                    .catch(error => {
                        console.error('Error loading tracks:', error);
                        showError('Failed to load tracks');
                    });
            }

            function displayTracks() {
                const tbody = document.getElementById('tracks-tbody');
                
                if (tracks.length === 0) {
                    tbody.innerHTML = `
                        <tr>
                            <td colspan="5" class="px-6 py-4 text-center text-gray-500">
                                No tracks found
                            </td>
                        </tr>
                    `;
                    return;
                }

                tbody.innerHTML = tracks.map(track => `
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900">${track.title}</div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
                                ${track.category ? track.category.name : 'Unknown'}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            ${formatDuration(track.duration_sec)}
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${track.published_at ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}">
                                ${track.published_at ? 'Published' : 'Draft'}
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <button onclick="editTrack(${track.id})" class="text-indigo-600 hover:text-indigo-900 mr-3">Edit</button>
                            <button onclick="deleteTrack(${track.id})" class="text-red-600 hover:text-red-900">Delete</button>
                        </td>
                    </tr>
                `).join('');
            }

            function formatDuration(seconds) {
                if (!seconds) return '0:00';
                const mins = Math.floor(seconds / 60);
                const secs = Math.floor(seconds % 60);
                return `${mins}:${secs.toString().padStart(2, '0')}`;
            }

            function showAddModal() {
                document.getElementById('modal-title').textContent = 'Add New Track';
                document.getElementById('track-form').reset();
                document.getElementById('track-id').value = '';
                showModal();
            }

            function showEditModal(track) {
                document.getElementById('modal-title').textContent = 'Edit Track';
                document.getElementById('track-id').value = track.id;
                document.getElementById('title').value = track.title;
                document.getElementById('category_id').value = track.category_id;
                document.getElementById('duration_sec').value = track.duration_sec;
                document.getElementById('audio_path').value = track.audio_path || '';
                document.getElementById('transcript_path').value = track.transcript_path || '';
                document.getElementById('tags').value = track.tags ? track.tags.join(', ') : '';
                document.getElementById('published_at').value = track.published_at ? track.published_at.slice(0, 16) : '';
                showModal();
            }

            function showModal() {
                document.getElementById('track-modal').classList.remove('hidden');
            }

            function hideModal() {
                document.getElementById('track-modal').classList.add('hidden');
            }

            function handleSubmit(e) {
                e.preventDefault();
                
                const formData = new FormData(e.target);
                const trackId = formData.get('id');
                const data = {
                    title: formData.get('title'),
                    category_id: parseInt(formData.get('category_id')),
                    duration_sec: parseInt(formData.get('duration_sec')),
                    audio_path: formData.get('audio_path'),
                    transcript_path: formData.get('transcript_path') || null,
                    tags: formData.get('tags') ? formData.get('tags').split(',').map(t => t.trim()).filter(t => t) : [],
                    published_at: formData.get('published_at') || null,
                };

                const url = trackId ? `/api/v1/admin/library/tracks/${trackId}` : '/api/v1/admin/library/tracks';
                const method = trackId ? 'PATCH' : 'POST';

                fetch(url, {
                    method: method,
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify(data)
                })
                .then(response => response.json())
                .then(result => {
                    hideModal();
                    loadTracks(); // Reload tracks
                    showSuccess(trackId ? 'Track updated successfully' : 'Track created successfully');
                })
                .catch(error => {
                    console.error('Error saving track:', error);
                    showError('Failed to save track');
                });
            }

            function editTrack(id) {
                const track = tracks.find(t => t.id === id);
                if (track) {
                    showEditModal(track);
                }
            }

            function deleteTrack(id) {
                if (!confirm('Are you sure you want to delete this track?')) return;

                fetch(`/api/v1/admin/library/tracks/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    }
                })
                .then(response => {
                    if (response.ok) {
                        loadTracks(); // Reload tracks
                        showSuccess('Track deleted successfully');
                    } else {
                        throw new Error('Failed to delete track');
                    }
                })
                .catch(error => {
                    console.error('Error deleting track:', error);
                    showError('Failed to delete track');
                });
            }

            function showSuccess(message) {
                // Simple success notification - could be enhanced with a toast
                alert(message);
            }

            function showError(message) {
                // Simple error notification - could be enhanced with a toast
                alert('Error: ' + message);
            }

            // Make functions globally available for onclick handlers
            window.editTrack = editTrack;
            window.deleteTrack = deleteTrack;
        });
    </script>
</x-app-layout>
