<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ClearAllReceiptData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:clear-all-receipt-data';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Deletes all receipts, associated data, files, and clears receipt-related jobs from queues.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        if (! $this->confirm('Are you sure you want to PERMANENTLY delete all receipts, their lines, matches, associated files, and queued jobs? This action cannot be undone.')) {
            $this->info('Operation cancelled.');

            return 1;
        }

        $this->info('Starting data clearing process...');

        // 1. Delete receipts and their files
        $this->info('Deleting receipts and their files from storage...');
        $receipts = Receipt::all();
        $progressBar = $this->output->createProgressBar($receipts->count());
        $progressBar->start();

        foreach ($receipts as $receipt) {
            if ($receipt->image_path && Storage::disk($receipt->image_disk)->exists($receipt->image_path)) {
                Storage::disk($receipt->image_disk)->delete($receipt->image_path);
            }
            $receipt->receipt_lines()->delete();
            $receipt->transaction_matches()->delete();
            $receipt->delete();
            $progressBar->advance();
        }
        $progressBar->finish();
        $this->newLine();
        $this->info('Receipts and files deleted.');

        // 2. Clear queues
        $this->info('Clearing job queues...');
        DB::table('jobs')->truncate();
        DB::table('failed_jobs')->truncate();
        DB::table('job_batches')->truncate();
        $this->info('Job queues cleared.');

        $this->info('All receipt data and jobs have been cleared successfully.');

        return 0;
    }
}
