<?php

namespace App\Http\Controllers;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Http\Resources\LibraryCategoryResource;
use App\Http\Resources\LibraryTrackResource;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class LibraryController extends Controller
{
    public function categories()
    {
        return LibraryCategoryResource::collection(LibraryCategory::orderBy('name')->get());
    }

    public function index(Request $request)
    {
        $q = LibraryTrack::query()->with('category');
        if ($request->filled('category_id')) {
            $q->where('category_id', $request->integer('category_id'));
        }
        if ($search = $request->string('q')->toString()) {
            $q->where(function ($w) use ($search) {
                $w->where('title', 'ilike', "%$search%")
                  ->orWhereJsonContains('tags', $search);
            });
        }
        $page = $q->paginate(15);
        return LibraryTrackResource::collection($page);
    }

    public function show(int $id)
    {
        $track = LibraryTrack::findOrFail($id);
        try {
            $disk = Storage::disk('gcs');
        } catch (\InvalidArgumentException $e) {
            $disk = Storage::disk('public');
        }

        $audioUrl = null;
        if ($track->audio_path) {
            try {
                $audioUrl = $disk->temporaryUrl($track->audio_path, now()->addMinutes(10));
            } catch (\Throwable $e) {
                $audioUrl = method_exists($disk, 'url') ? $disk->url($track->audio_path) : $track->audio_path;
            }
        }

        $transcriptUrl = null;
        if ($track->transcript_path) {
            try {
                $transcriptUrl = $disk->temporaryUrl($track->transcript_path, now()->addMinutes(10));
            } catch (\Throwable $e) {
                $transcriptUrl = method_exists($disk, 'url') ? $disk->url($track->transcript_path) : $track->transcript_path;
            }
        }
        return response()->json([
            'id' => $track->id,
            'title' => $track->title,
            'duration_sec' => $track->duration_sec,
            'audio_url' => $audioUrl,
            'transcript_url' => $transcriptUrl,
            'tags' => $track->tags,
            'published_at' => optional($track->published_at)->toISOString(),
        ]);
    }
}


