<?php

namespace App\Http\Controllers;

use App\Http\Resources\LibraryCategoryResource;
use App\Http\Resources\LibraryTrackResource;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class LibraryController extends Controller
{
    /**
     * Get all library categories
     */
    public function categories(): JsonResponse
    {
        $categories = LibraryCategory::orderBy('name')->get();
        return LibraryCategoryResource::collection($categories)->response();
    }

    /**
     * Get library tracks with filtering and pagination
     */
    public function index(Request $request): JsonResponse
    {
        // Validate request parameters
        $validated = $request->validate([
            'category_id' => 'nullable|integer|exists:library_categories,id',
            'q' => 'nullable|string|max:255',
            'duration' => 'nullable|string|in:0-10,10-20,20+',
            'per_page' => 'nullable|integer|min:1|max:100',
        ]);

        $query = LibraryTrack::query()->with('category');
        
        // Category filter
        if (isset($validated['category_id'])) {
            $query->where('category_id', $validated['category_id']);
        }
        
        // Search filter
        if (!empty($validated['q'])) {
            $search = $validated['q'];
            $query->where(function ($q) use ($search) {
                $q->where('title', 'ilike', "%{$search}%")
                    ->orWhereJsonContains('tags', $search)
                    ->orWhere('description', 'ilike', "%{$search}%");
            });
        }
        
        // Duration filter
        if (isset($validated['duration'])) {
            switch ($validated['duration']) {
                case '0-10':
                    $query->where('duration_sec', '<=', 600); // 10 minutes = 600 seconds
                    break;
                case '10-20':
                    $query->where('duration_sec', '>', 600)->where('duration_sec', '<=', 1200);
                    break;
                case '20+':
                    $query->where('duration_sec', '>', 1200);
                    break;
            }
        }
        
        // Pagination
        $perPage = $validated['per_page'] ?? 15;
        $tracks = $query->orderBy('published_at', 'desc')->paginate($perPage);

        return LibraryTrackResource::collection($tracks)->response();
    }

    /**
     * Get a specific library track
     */
    public function show(int $id): JsonResponse
    {
        $track = LibraryTrack::with('category')->findOrFail($id);
        
        return (new LibraryTrackResource($track))->response();
    }
}
