<?php

namespace App\Http\Controllers;

use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use App\Http\Resources\LibraryCategoryResource;
use App\Http\Resources\LibraryTrackResource;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class LibraryController extends Controller
{
    public function categories()
    {
        return LibraryCategoryResource::collection(LibraryCategory::orderBy('name')->get());
    }

    public function index(Request $request)
    {
        $q = LibraryTrack::query()->with('category');
        if ($request->filled('category_id')) {
            $q->where('category_id', $request->integer('category_id'));
        }
        if ($search = $request->string('q')->toString()) {
            $q->where(function ($w) use ($search) {
                $w->where('title', 'ilike', "%$search%")
                  ->orWhereJsonContains('tags', $search);
            });
        }
        $page = $q->paginate(15);
        return response()->json([
            'data' => LibraryTrackResource::collection($page->items()),
            'meta' => [
                'current_page' => $page->currentPage(),
                'per_page' => $page->perPage(),
                'total' => $page->total(),
                'last_page' => $page->lastPage(),
            ],
            'links' => [
                'first' => $page->url(1),
                'prev' => $page->previousPageUrl(),
                'next' => $page->nextPageUrl(),
                'last' => $page->url($page->lastPage()),
            ],
        ]);
    }

    public function show(int $id)
    {
        $track = LibraryTrack::findOrFail($id);
        $disk = Storage::disk('gcs');
        $audioUrl = $disk->temporaryUrl($track->audio_path, now()->addMinutes(10));
        $transcriptUrl = $track->transcript_path ? $disk->temporaryUrl($track->transcript_path, now()->addMinutes(10)) : null;
        return response()->json([
            'id' => $track->id,
            'title' => $track->title,
            'duration_sec' => $track->duration_sec,
            'audio_url' => $audioUrl,
            'transcript_url' => $transcriptUrl,
            'tags' => $track->tags,
            'published_at' => optional($track->published_at)->toISOString(),
        ]);
    }
}


