<?php

namespace App\Http\Controllers;

use App\Http\Resources\LibraryCategoryResource;
use App\Http\Resources\LibraryTrackResource;
use App\Models\LibraryCategory;
use App\Models\LibraryTrack;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class LibraryController extends Controller
{
    public function categories()
    {
        // Return plain array for ease of use in dev UI; can wrap in resource if needed
        return response()->json([
            'data' => LibraryCategory::orderBy('name')->get(['id', 'name', 'slug']),
        ]);
    }

    public function index(Request $request)
    {
        $q = LibraryTrack::query()->with('category');
        
        // Category filter
        if ($request->filled('category_id')) {
            $q->where('category_id', $request->integer('category_id'));
        }
        
        // Search filter
        if ($search = $request->string('q')->toString()) {
            $q->where(function ($w) use ($search) {
                $w->where('title', 'ilike', "%$search%")
                    ->orWhereJsonContains('tags', $search)
                    ->orWhere('description', 'ilike', "%$search%");
            });
        }
        
        // Duration filter
        if ($duration = $request->string('duration')->toString()) {
            switch ($duration) {
                case '0-10':
                    $q->where('duration_sec', '<=', 600); // 10 minutes = 600 seconds
                    break;
                case '10-20':
                    $q->where('duration_sec', '>', 600)->where('duration_sec', '<=', 1200);
                    break;
                case '20+':
                    $q->where('duration_sec', '>', 1200);
                    break;
            }
        }
        
        // Pagination
        $perPage = $request->integer('per_page', 15);
        $page = $q->orderBy('published_at', 'desc')->paginate($perPage);

        return LibraryTrackResource::collection($page);
    }

    public function show(int $id)
    {
        $track = LibraryTrack::findOrFail($id);
        try {
            $disk = Storage::disk('gcs');
        } catch (\InvalidArgumentException $e) {
            $disk = Storage::disk('public');
        }

        $audioUrl = null;
        if ($track->audio_path) {
            try {
                $audioUrl = $disk->temporaryUrl($track->audio_path, now()->addMinutes(10));
            } catch (\Throwable $e) {
                $audioUrl = method_exists($disk, 'url') ? $disk->url($track->audio_path) : $track->audio_path;
            }
        }

        $transcriptUrl = null;
        if ($track->transcript_path) {
            try {
                $transcriptUrl = $disk->temporaryUrl($track->transcript_path, now()->addMinutes(10));
            } catch (\Throwable $e) {
                $transcriptUrl = method_exists($disk, 'url') ? $disk->url($track->transcript_path) : $track->transcript_path;
            }
        }

        return response()->json([
            'data' => [
                'id' => $track->id,
                'title' => $track->title,
                'description' => $track->description,
                'duration_sec' => $track->duration_sec,
                'audio_url' => $audioUrl,
                'transcript_url' => $transcriptUrl,
                'tags' => $track->tags,
                'category' => $track->category ? [
                    'id' => $track->category->id,
                    'name' => $track->category->name,
                    'slug' => $track->category->slug,
                ] : null,
                'published_at' => optional($track->published_at)->toISOString(),
            ],
        ]);
    }
}
