<?php

namespace App\Services\OCR;

use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class ReceiptOCRService
{
    public function __construct(private VertexClient $vertex) {}

    public function extractReceiptData(Receipt $receipt): void
    {
        // Get the image path
        $imagePath = Storage::disk($receipt->image_disk)->path($receipt->image_path);
        
        if (!file_exists($imagePath)) {
            return;
        }

        // Use Vertex AI to extract receipt data from the image
        $model = config('vertex.models.ocr', 'gemini-2.5-flash-lite');
        $system = 'You are an expert at extracting comprehensive data from receipt images. Extract ALL available information to help with future analysis and matching.

REQUIRED FIELDS:
- merchant_name: Business name
- receipt_date: Date in YYYY-MM-DD format
- total_amount: Total amount as decimal (e.g., 12.50)
- currency: Currency code (e.g., GBP, USD, EUR)
- lines: Array of line items with description, quantity, unit_price, line_total

OPTIONAL FIELDS (extract if visible):
- receipt_number: Receipt/invoice number
- cashier_name: Cashier or staff member name
- store_location: Store address or location
- phone_number: Business phone number
- email: Business email
- website: Business website
- vat_number: VAT/tax registration number
- payment_method: Cash, card, etc.
- subtotal: Amount before tax
- tax_amount: Tax/VAT amount
- discount_amount: Any discounts applied
- tip_amount: Tip or service charge
- transaction_id: Transaction reference number
- loyalty_points: Loyalty points earned
- opening_hours: Store hours if visible
- return_policy: Return policy text if visible
- additional_notes: Any other relevant text

For line items, extract:
- description: Item description
- quantity: Number of items
- unit_price: Price per unit
- line_total: Total for this line
- category: Product category if identifiable (food, fuel, clothing, etc.)
- brand: Brand name if visible
- barcode: Barcode number if visible
- tax_rate: Tax rate for this item if shown

Output strict JSON with all available fields. Be thorough - extract every piece of text and data visible on the receipt.';
        
        // Read image as base64
        $imageData = base64_encode(file_get_contents($imagePath));
        
        $prompt = [
            'image' => $imageData,
            'instructions' => 'Extract all receipt data from this image'
        ];

        $response = $this->vertex->generate($model, json_encode($prompt), $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            // Fallback to sample data if OCR fails
            $data = [
                'merchant_name' => 'Unknown Merchant ' . $receipt->id,
                'receipt_date' => now()->format('Y-m-d'),
                'total_amount' => 0,
                'currency' => 'GBP',
                'lines' => []
            ];
        } else {
            $data = $response['json'];
        }

        // Update receipt with extracted data
        if (!empty($data['merchant_name'])) {
            $receipt->merchant_name = $data['merchant_name'];
        }

        if (!empty($data['receipt_date'])) {
            try {
                // Parse the date and convert to Y-m-d format
                $date = Carbon::parse($data['receipt_date']);
                $receipt->receipt_date = $date->format('Y-m-d');
            } catch (\Exception $e) {
                // If date parsing fails, keep the original date
            }
        }

        if (!empty($data['total_amount'])) {
            $receipt->total_amount = (float) $data['total_amount'];
        }

        if (!empty($data['currency'])) {
            $receipt->currency = strtoupper($data['currency']);
        }

        $receipt->save();

        // Process line items
        if (!empty($data['lines']) && is_array($data['lines'])) {
            // Clear existing lines
            $receipt->lines()->delete();
            
            foreach ($data['lines'] as $lineData) {
                if (empty($lineData['description'])) {
                    continue;
                }

                ReceiptLine::create([
                    'receipt_id' => $receipt->id,
                    'description' => $lineData['description'],
                    'quantity' => (float) ($lineData['quantity'] ?? 1),
                    'unit_price' => (float) ($lineData['unit_price'] ?? 0),
                    'line_total' => (float) ($lineData['line_total'] ?? 0),
                ]);
            }
        }
    }
}
