<?php

namespace App\Services\OCR;

use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class ReceiptOCRService
{
    public function __construct(private VertexClient $vertex) {}

    public function extractReceiptData(Receipt $receipt): void
    {
        // Get the image path
        $imagePath = Storage::disk($receipt->image_disk)->path($receipt->image_path);
        
        if (!file_exists($imagePath)) {
            return;
        }

        // Use Vertex AI to extract receipt data from the image
        $model = config('vertex.models.ocr', 'gemini-1.5-flash');
        $system = 'You are an expert at extracting data from receipt images. Extract the merchant name, receipt date, total amount, currency, and line items. For dates, use YYYY-MM-DD format. For amounts, use decimal format (e.g., 12.50). Output strict JSON with keys: merchant_name, receipt_date, total_amount, currency, lines (array of objects with description, quantity, unit_price, line_total).';
        
        // Read image as base64
        $imageData = base64_encode(file_get_contents($imagePath));
        
        $prompt = [
            'image' => $imageData,
            'instructions' => 'Extract all receipt data from this image'
        ];

        $response = $this->vertex->generate($model, json_encode($prompt), $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            return;
        }

        $data = $response['json'];

        // Update receipt with extracted data
        if (!empty($data['merchant_name'])) {
            $receipt->merchant_name = $data['merchant_name'];
        }

        if (!empty($data['receipt_date'])) {
            try {
                // Parse the date and convert to Y-m-d format
                $date = Carbon::parse($data['receipt_date']);
                $receipt->receipt_date = $date->format('Y-m-d');
            } catch (\Exception $e) {
                // If date parsing fails, keep the original date
            }
        }

        if (!empty($data['total_amount'])) {
            $receipt->total_amount = (float) $data['total_amount'];
        }

        if (!empty($data['currency'])) {
            $receipt->currency = strtoupper($data['currency']);
        }

        $receipt->save();

        // Process line items
        if (!empty($data['lines']) && is_array($data['lines'])) {
            // Clear existing lines
            $receipt->lines()->delete();
            
            foreach ($data['lines'] as $lineData) {
                if (empty($lineData['description'])) {
                    continue;
                }

                ReceiptLine::create([
                    'receipt_id' => $receipt->id,
                    'description' => $lineData['description'],
                    'quantity' => (float) ($lineData['quantity'] ?? 1),
                    'unit_price' => (float) ($lineData['unit_price'] ?? 0),
                    'line_total' => (float) ($lineData['line_total'] ?? 0),
                ]);
            }
        }
    }
}
