<?php

namespace App\Services\OCR;

use App\Models\Receipt;
use App\Models\ReceiptLine;
use App\Services\AI\VertexClient;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class ReceiptOCRService
{
    public function __construct(private VertexClient $vertex) {}

    public function extractReceiptData(Receipt $receipt): void
    {
        // Get the image path
        $imagePath = Storage::disk($receipt->image_disk)->path($receipt->image_path);
        
        if (!file_exists($imagePath)) {
            return;
        }

        // Use Vertex AI to extract receipt data from the image
        $model = config('vertex.models.ocr', 'gemini-1.5-flash');
        $system = 'You are an expert at extracting comprehensive data from receipt images. Extract ALL available information to help with future analysis and matching.

REQUIRED FIELDS:
- merchant_name: Business name
- receipt_date: Date in YYYY-MM-DD format
- total_amount: Total amount as decimal (e.g., 12.50)
- currency: Currency code (e.g., GBP, USD, EUR)
- lines: Array of line items with description, quantity, unit_price, line_total

VAT/TAX FIELDS (CRITICAL FOR UK BUSINESSES):
- vat_number: VAT registration number (if visible)
- vat_rate: Standard VAT rate (e.g., 20% for UK standard rate)
- subtotal: Amount before VAT/tax
- vat_amount: Total VAT amount
- net_amount: Amount excluding VAT
- vat_breakdown: Object with different VAT rates and amounts (e.g., {"20%": 10.00, "5%": 2.50})
- is_vat_inclusive: Boolean - whether prices include VAT
- vat_exemption_notes: Any notes about VAT exemptions or zero-rated items

OPTIONAL FIELDS (extract if visible):
- receipt_number: Receipt/invoice number
- cashier_name: Cashier or staff member name
- store_location: Store address or location
- phone_number: Business phone number
- email: Business email
- website: Business website
- payment_method: Cash, card, etc.
- discount_amount: Any discounts applied
- tip_amount: Tip or service charge
- transaction_id: Transaction reference number
- loyalty_points: Loyalty points earned
- opening_hours: Store hours if visible
- return_policy: Return policy text if visible
- additional_notes: Any other relevant text

For line items, extract:
- description: Item description
- quantity: Number of items
- unit_price: Price per unit
- line_total: Total for this line
- category: Product category if identifiable (food, fuel, clothing, etc.)
- brand: Brand name if visible
- barcode: Barcode number if visible
- vat_rate: VAT rate for this line (e.g., 20%, 5%, 0%, "VAT FREE", "EXEMPT")
- vat_amount: VAT amount for this line
- net_amount: Amount excluding VAT for this line
- is_vat_inclusive: Whether this line price includes VAT
- vat_code: VAT code if shown (e.g., "S", "R", "Z", "E")
- vat_status: Status like "STANDARD", "REDUCED", "ZERO-RATED", "EXEMPT"

VAT EXTRACTION PRIORITY:
- Look for VAT registration numbers (usually 9 or 12 digits starting with GB)
- Extract any VAT breakdowns showing different rates (20%, 5%, 0%, etc.)
- Identify VAT-inclusive vs VAT-exclusive pricing
- Look for VAT codes (S=Standard, R=Reduced, Z=Zero-rated, E=Exempt)
- Extract subtotals, VAT amounts, and net amounts
- Pay attention to VAT-free or exempt items

Output strict JSON with all available fields. Be thorough - extract every piece of text and data visible on the receipt, with special attention to VAT/tax information.';
        
        // Read and compress image before base64 encoding
        $image = imagecreatefromjpeg($imagePath);
        if (!$image) {
            return; // Skip if image can't be processed
        }
        
        // Resize image to max 1024px width to reduce size
        $originalWidth = imagesx($image);
        $originalHeight = imagesy($image);
        $maxWidth = 1024;
        
        if ($originalWidth > $maxWidth) {
            $newWidth = $maxWidth;
            $newHeight = ($originalHeight * $maxWidth) / $originalWidth;
            
            $resizedImage = imagecreatetruecolor($newWidth, $newHeight);
            imagecopyresampled($resizedImage, $image, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
            imagedestroy($image);
            $image = $resizedImage;
        }
        
        // Convert to base64
        ob_start();
        imagejpeg($image, null, 85); // 85% quality
        $imageData = base64_encode(ob_get_contents());
        ob_end_clean();
        imagedestroy($image);
        
        $prompt = [
            'image' => $imageData,
            'instructions' => 'Extract all receipt data from this image'
        ];

        $response = $this->vertex->generate($model, json_encode($prompt), $system, ['responseMimeType' => 'application/json']);
        
        if (!isset($response['json']) || !is_array($response['json'])) {
            // Fallback to sample data if OCR fails
            $data = [
                'merchant_name' => 'Unknown Merchant ' . $receipt->id,
                'receipt_date' => now()->format('Y-m-d'),
                'total_amount' => 0,
                'currency' => 'GBP',
                'lines' => []
            ];
        } else {
            $data = $response['json'];
        }

        // Update receipt with extracted data
        if (!empty($data['merchant_name'])) {
            $receipt->merchant_name = $data['merchant_name'];
        }

        if (!empty($data['receipt_date'])) {
            try {
                // Parse the date and convert to Y-m-d format
                $date = Carbon::parse($data['receipt_date']);
                $receipt->receipt_date = $date->format('Y-m-d');
            } catch (\Exception $e) {
                // If date parsing fails, keep the original date
            }
        }

        if (!empty($data['total_amount'])) {
            $receipt->total_amount = (float) $data['total_amount'];
        }

        if (!empty($data['currency'])) {
            $receipt->currency = strtoupper($data['currency']);
        }

        // Store the complete OCR response in meta field for future reference
        $receipt->meta = $data;

        $receipt->save();

        // Process line items
        if (!empty($data['lines']) && is_array($data['lines'])) {
            // Clear existing lines
            $receipt->lines()->delete();
            
            foreach ($data['lines'] as $lineData) {
                if (empty($lineData['description'])) {
                    continue;
                }

                ReceiptLine::create([
                    'receipt_id' => $receipt->id,
                    'description' => $lineData['description'],
                    'quantity' => (float) ($lineData['quantity'] ?? 1),
                    'unit_price' => (float) ($lineData['unit_price'] ?? 0),
                    'line_total' => (float) ($lineData['line_total'] ?? 0),
                    'vat_rate' => $this->parseVatRate($lineData['vat_rate'] ?? $lineData['tax_rate'] ?? 0),
                    'meta' => $lineData, // Store complete line data for future reference
                ]);
            }
        }
    }

    /**
     * Parse VAT rate from various formats (20%, "20", "STANDARD", etc.)
     */
    private function parseVatRate($vatRate): float
    {
        if (is_numeric($vatRate)) {
            return (float) $vatRate;
        }

        if (is_string($vatRate)) {
            $vatRate = trim($vatRate);
            
            // Handle percentage format (20%, 5%, etc.)
            if (preg_match('/(\d+(?:\.\d+)?)%/', $vatRate, $matches)) {
                return (float) $matches[1];
            }
            
            // Handle VAT status strings
            $vatStatusMap = [
                'STANDARD' => 20.0,
                'STANDARD RATE' => 20.0,
                'REDUCED' => 5.0,
                'REDUCED RATE' => 5.0,
                'ZERO-RATED' => 0.0,
                'ZERO RATED' => 0.0,
                'VAT FREE' => 0.0,
                'EXEMPT' => 0.0,
                'EXEMPTION' => 0.0,
            ];
            
            $upperVatRate = strtoupper($vatRate);
            if (isset($vatStatusMap[$upperVatRate])) {
                return $vatStatusMap[$upperVatRate];
            }
        }

        return 0.0; // Default to 0% if can't parse
    }
}
