<?php

namespace App\Http\Livewire\Production\Reports;

use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Shipment;
use App\Models\Suppliers;
use App\Models\Departments;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use App\Models\ShipmentLine;
use Livewire\WithPagination;
use App\Livewire\Forms\CPForm;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Livewire\Forms\AsosCPForm;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class CriticalPath extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    protected function filterKey(): string
    {
        return 'whcriticalpathfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['view', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'columns', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }
    public function updated($propertyName)
    {
        if(in_array($propertyName, $this->filters())){
            // Validate individual field
            $this->validateOnly($propertyName);

            // Check for errors and reset if validation fails
            if ($this->getErrorBag()->has($propertyName)) {
                $this->reset($propertyName);
            } else {
                // Reset pagination only if a relevant property is changed
                if (in_array($propertyName, ['view', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search', 'paginate'])) {
                    $this->resetPage();
                }

                // Save filter settings since validation passed
                $this->saveFilterSettings();
            }
        }
    }


    public CPForm $form;

    #[Validate([
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
        'view' => ['required', 'in:unshipped,financeNC,all'],
        'department' => ['nullable', 'exists:departments,id'],
        'customer' => ['required', 'exists:customers,id'],
        'season' => ['required', 'exists:seasons,id'],
        'factory' => ['nullable', 'exists:suppliers,id'],
        'coo' => ['nullable', 'exists:countries,id'],
        'category' => ['nullable', 'in:ladies,mens,accessories,childrens'],
        'search' => ['nullable', 'string', 'max:30'],
        'selected' => ['array'],
        'selected.*' => ['nullable|boolean'],
    ])]
    public $paginate = 21, $view = "unshipped", $department = NULL, $customer = 1, $season = 1, $factory = "", $coo, $category = NULL, $search = "", $selected = [];
    private $sl;
    public $sizes;

    public $columns = [
        'Factory' => TRUE,
        'Image' => TRUE,
        'RT Number' => TRUE,
        'RT Description' => TRUE,
        'PO Number' => TRUE,
        'Customer Reference' => TRUE,
        'Colour' => TRUE,
        'Sell Price' => TRUE,
        'Total Qty' => TRUE,
        'Warehouse' => TRUE,
        'Order Date' => TRUE,
        'Sent to Factory' => TRUE,
        'Shipment Complete' => TRUE,

        'Yarn' => TRUE,
        'Yarn Colour' => TRUE,
        'Factory Order Yarn (FOY)' => TRUE,
        'PRODs' => TRUE,

        'Buttons' => TRUE,
        'Labels' => TRUE,
        'Care' => TRUE,
        'Pre Production Comments' => TRUE,
        'Photoshoot Deadline' => TRUE,
        'Sealer Deadline' => TRUE,
        'Sealer Sample' => TRUE,
        'Start Knit' => TRUE,
        'Shipment Sample' => TRUE,
        'Exfty' => TRUE,
        'Revised Exfty' => TRUE,
        'Customer Exfty' => TRUE,
        'Comments' => TRUE,
    ];

    public function updatedForm($val, $var){
        $this->form->save($val, $var);
    }

    public function go(){

    }

    public function clearFilters(){
        $this->reset('paginate', 'view', 'department', 'customer', 'season', 'factory', 'coo', 'category', 'search');
        $this->updatedPaginate();
        $this->updatedSeason();
        $this->updatedDepartment();
        $this->updatedCustomer();
        $this->updatedFactory();
        $this->updatedCoo();
        $this->updatedCategory();
        $this->updatedSearch();
    }

    #[On('close-modal')]
	public function modalClosed(){
		$this->render();
	}

    public function mount()
    {
        $this->loadFilterSettings();

        $this->sl = $this->getData();
        $this->form->mount($this->sl->toArray());
        $this->sizes = $this->sl->pluck('sizes')->collapse()->unique('size')->pluck('size', 'sizes_id');
    }

    public function render()
    {
        if (empty($this->sl)) {
            $this->mount();
        }

        return view('livewire.production.reports.critical-path', ['sl' => $this->sl]);
    }

    public function getStyleCount(Styles $style){
        return $style->style_versions->flatMap(function ($version) {
            return $version->colourways->flatMap(function ($colourway) {
                return $colourway->customer_order_lines;
            });
        })->count();
    }


    public function edit(){
		$this->dispatch('edit-cp', selected: json_encode(array_keys($this->selected, TRUE)));
	}

    public function getData(){
        $query = ShipmentLine::
        select(
            'shipment_lines.*',
            'factories.name as factory',
            'colourways.img_thumb',
            'colourways.id as colourways_id',
            'styles.designs_id as designs_id',
            'customer_orders.customer_po',
            'customer_orders.id as customer_orders_id',
            'styles.customer_ref',
            'styles.id as styles_id',
            'designs.description',
            'designs.id as designs_id',
            'colourways.name as colourways_name',
            'customers.currency as customer_currency',
            'factories.currency as factory_currency',
            'shipment_lines.exfty',
            'customer_addresses.name as customer_address',

            'colourways.customer_description',
            'customer_orders.order_date as order_date',
            'customer_order_lines.id as customer_order_lines_id',
            'customer_order_lines.order_sent_factory',
            'colourways.composition',
            'colourways.colour_approval_comments',

            'customer_order_lines.bulk_yarn_order_date',
            'customer_order_lines.bulk_yarn_order_comments',
            'customer_order_lines.bulk_yarn_order_delivery_date',
            'customer_order_lines.bulk_yarn_order_delivery_comments',

            'customer_order_lines.start_knit',
            'shipment_lines.start_knit_comments',
            'styles.customer_samp_no',

            'styles.green_seal_approval_comments',
            'styles.photo_shoot_sample_comments',

            'colourways.accessories',
            'colourways.testing_comments',



            'shipment_lines.complete',
            'shipment_lines.id as shipment_lines_id',
            'shipment_lines.yarn_comments',

            'shipment_lines.buttons_required',
            'shipment_lines.buttons',
            'shipment_lines.buttons_done',
            'shipment_lines.labels',
            'shipment_lines.labels_done',
            'shipment_lines.care',
            'shipment_lines.care_done',
            'shipment_lines.factory_order_yarn',

            'customer_order_lines.factory_cust_date',

            'shipment_lines.cp_notes',
            'shipment_lines.pre_production_comments',


            'prices.id as prices_id',
            'prices.discount',
            )
            ->selectRaw("
            COALESCE(
                (
                    SELECT
                        JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                    FROM audits AS audit
                    WHERE audit.auditable_id = shipment_lines.id
                    AND audit.auditable_type = 'App\\\Models\\\ShipmentLine'
                    AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                    ORDER BY audit.created_at ASC
                    LIMIT 1
                ),
                shipment_lines.exfty
            ) AS first_exfty
            ")
            // ->selectRaw("CASE WHEN(shipment_lines.factory_order_yarn = 1) THEN 'true' ELSE 'false' END as factory_order_yarn")
            ->selectRaw("FORMAT(prices.cmt,2) as cmt")
            ->selectRaw("FORMAT(prices.quote,2) as quote")

            ->selectRaw("(SELECT CONCAT('[', GROUP_CONCAT(JSON_OBJECT('id', yc.id, 'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_id', yc.id, 'yarn_id', y.id, 'prods', (SELECT CONCAT('[', GROUP_CONCAT(JSON_OBJECT('prod', yo.id,'ordered', yo.date, 'actual_ship_date', yol.actual_ship_date, 'delivered', yol.delivery_date, 'ship_date', yol.ship_date)), ']') FROM yarn_orders yo JOIN yarn_order_lines yol ON yol.yarn_order_id = yo.id AND yol.yarn_colours_id = cy.yarn_colours_id JOIN yarn_order_line_styles yols ON yols.yarn_order_lines_id = yol.id AND yols.customer_order_lines_id = customer_order_lines.id))), ']') FROM colourway_yarns cy JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id JOIN yarns y ON y.id = yc.yarn_id JOIN counts ON counts.id = y.counts_id WHERE cy.colourways_id = colourways.id) as yarns")
            ->selectRaw("
            COALESCE(
                (
                    SELECT
                        json_object(
                            'status', s.status,
                            'id', s.id,
                            'comments', s.comments,
                            'date_sent', s.date_sent,
                            'note', NULL
                        ) AS sample
                    FROM
                        samples s
                    WHERE
                        s.colourways_id = colourways.id
                        AND s.sample_types_id = 7
                    ORDER BY
                        s.created_at DESC
                    LIMIT 1
                ),
                CASE
                    WHEN shipment_lines.shipment_status IS NOT NULL THEN
                        json_object(
                            'status', shipment_lines.shipment_status,
                            'id', NULL,
                            'comments', shipment_lines.sealer_comments,
                            'date_sent', NULL,
                            'note', '(SS)'
                        )
                END,
                CASE
                    WHEN CAST(json_extract(json_extract(customers.settings, '$.\"carryover-samples-required\"'), '$.\"7\"') AS UNSIGNED) = 1 THEN
                        json_object(
                            'status', 'Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                END,
                CASE
                    WHEN styles.carryover = 1 THEN
                        json_object(
                            'status', 'Not Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', '(Carryover)'
                        )
                END,
                CASE
                    WHEN CAST(json_extract(customers.samples_required, '$.\"7\"') AS UNSIGNED) = 1 THEN
                        json_object(
                            'status', 'Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                    ELSE
                        json_object(
                            'status', 'Not Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                END
            ) AS shipment
        ")
        ->selectRaw("
            COALESCE(
                (
                    SELECT
                        json_object(
                            'status', s.status,
                            'id', s.id,
                            'comments', s.comments,
                            'date_sent', s.date_sent,
                            'note', CASE WHEN s.colourways_id != colourways.id THEN c.name ELSE NULL END
                        ) AS sample
                    FROM
                        style_versions sv
                    INNER JOIN colourways c ON sv.id = c.style_versions_id
                    INNER JOIN samples s ON s.colourways_id = c.id AND s.sample_types_id = 3
                    WHERE
                        sv.id = style_versions.id
                    ORDER BY
                        CASE WHEN s.colourways_id = colourways.id THEN 0 ELSE 1 END,
                        s.created_at DESC
                    LIMIT 1
                ),
                CASE
                    WHEN shipment_lines.sealer_status IS NOT NULL THEN
                        json_object(
                            'status', shipment_lines.sealer_status,
                            'id', NULL,
                            'comments', shipment_lines.sealer_comments,
                            'date_sent', NULL,
                            'note', '(SS)'
                        )
                END,
                CASE
                    WHEN CAST(json_extract(json_extract(customers.settings, '$.\"carryover-samples-required\"'), '$.\"3\"') AS UNSIGNED) = 1 THEN
                        json_object(
                            'status', 'Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                END,
                CASE
                    WHEN styles.carryover = 1 THEN
                        json_object(
                            'status', 'Not Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', '(Carryover)'
                        )
                END,
                CASE
                    WHEN CAST(json_extract(customers.samples_required, '$.\"3\"') AS UNSIGNED) = 1 THEN
                        json_object(
                            'status', 'Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                    ELSE
                        json_object(
                            'status', 'Not Required',
                            'id', NULL,
                            'comments', NULL,
                            'date_sent', NULL,
                            'note', NULL
                        )
                END
            ) AS sealer

        ")

            ->selectRaw("
            (
                select
                concat(
                    '[',
                    group_concat(
                    JSON_OBJECT(
                        'colq_id', colq.id,
                        'sls_id', sls.id,
                        'size', sizes.name,
                        'sizes_id', sizes.id,
                        'qty', sls.qty,
                        'shipped_qty', sls.shipped_qty,
                        'quote_cache', colq.quote_cache
                    ) SEPARATOR ','
                    ),
                    ']'
                )
                from  shipment_line_sizes sls
                join sizes on sizes.id = sls.sizes_id and sls.shipment_line_id = shipment_lines.id
                join customer_order_line_quantities colq on colq.customer_order_lines_id = customer_order_lines.id and colq.sizes_id = sls.sizes_id
                left join price_resolutions pr on pr.style_versions_id = colourways.style_versions_id 
                    and pr.colourways_id = colourways.id 
                    and pr.sizes_id = sls.sizes_id 
                    and pr.phase_id is null 
                    and pr.season_id = customer_orders.seasons_id
                    and pr.fresh_at is not null
                group by sls.shipment_line_id
            ) AS sizes
        ")


        ->selectRaw('COUNT(*) OVER (PARTITION BY colourways.id, shipment_lines.exfty) AS colourway_count')
        ->selectRaw('FIRST_VALUE(shipment_lines.id) OVER (PARTITION BY colourways.id, shipment_lines.exfty) AS colourway_first')
        ->selectRaw('LAST_VALUE(shipment_lines.id) OVER (PARTITION BY colourways.id, shipment_lines.exfty) AS colourway_last')

        // ->whereRelation('customer_order_lines.customer_orders', 'customers_id', '=', '6')

        ->Join('customer_order_lines', function($join)
        {
            $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id');
            $join->on('customer_order_lines.cancelled', '=', DB::raw(0));
        })
        ->Join('customer_orders', function($join)
        {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customer_addresses', 'customer_addresses.id', '=', 'customer_orders.customer_addresses_id')
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('suppliers as factories', 'factories.id', '=', 'style_versions.factory_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->leftJoin(DB::raw('(
            SELECT style_versions_id, MAX(prices.id) as id, MAX(cmt) as cmt, MAX(quote) as quote, MAX(discount) as discount, MAX(updated_at) AS max_updated_at
            FROM prices
            GROUP BY style_versions_id
          ) AS prices'), function ($join) {
            $join->on('prices.style_versions_id', '=', 'style_versions.id');
          });

        if($this->view == "unshipped"){
			$query->where('shipment_lines.complete', FALSE);
		}
		if($this->view == "financeNC"){
			$query->where('shipment_lines.invoiced', FALSE);
		}

        // $query->whereIn('customer_orders.departments_id', Auth::user()->user_departments->pluck('departments_id'));

		$query->where('customer_orders.seasons_id', $this->season);
		$query->where('customer_orders.customers_id', $this->customer);
		if(!empty($this->factory)){
			$query->where('style_versions.factory_id', $this->factory);
		}
		if(!empty($this->coo)){
			$query->where('factories.coo', $this->coo);
		}
		if(!empty($this->department)){
			$query->where('customer_orders.departments_id', $this->department);
		}
		if(!empty($this->category)){
			$query->where('styles.category', $this->category);
		}

        if(!empty($this->search)){
            $query->search($this->search);
            // $query->where(function($query) {
            //     $query->where('designs.description', 'like', "%{$this->search}%")
            //     ->orwhere('styles.customer_ref', 'like', "%{$this->search}%")
            //     ->orwhere('designs.id', 'like', "%{$this->search}%")
            //     ->orwhere('customer_orders.customer_po', 'like', "%{$this->search}%")
            //     ->orwhere('colourways.name', 'like', "%{$this->search}%");
            // });
		}

        $query->orderBy('shipment_lines.exfty', 'asc')
        ->orderBy('styles.id', 'asc')
        ->orderBy('style_versions.factory_id', 'asc')
        ->orderBy('colourways.id', 'asc');
        $sl = $query->get();

        foreach($sl as $l=>$sline){
            $sline->rowspanStyle =
                $sl ->where('exfty', $sline->exfty)
                    ->where('styles_id', $sline->styles_id)->count();
                if($sline->sizes){
                    $sl[$l]->sizes = array_column(json_decode($sline->sizes ?? "", true), null, 'sizes_id');
                }
                $yarns = json_decode($sline->yarns);
                foreach($yarns ?? [] as $y=>$yarn){
                    if(!empty($yarn->prods)){
                        $yarn->prods = collect(json_decode($yarn->prods));
                        $sl[$l]->has_prods = 1;
                    }
                    elseif($sl[$l]->has_prods == 0){
                        $sl[$l]->has_prods = 0;
                    }
                }
                $sl[$l]->yarns = $yarns;

                $sl[$l]->factory_order_yarn = (bool) $sline->factory_order_yarn;
                $sl[$l]->buttons_done = (bool) $sline->buttons_done;
                $sl[$l]->labels_done = (bool) $sline->labels_done;
                $sl[$l]->care_done = (bool) $sline->care_done;
                $sl[$l]->complete = (bool) $sline->complete;

                $sl[$l]->foy_ordered = (bool) $sline->foy_ordered;
                $sl[$l]->foy_delivered = (bool) $sline->foy_delivered;

                $sl[$l]->buttons_required = (bool) $sline->buttons_required;

                $sl[$l]->description = Str::title($sline->description);
                $sl[$l]->colourways_name = Str::title($sline->colourways_name);
                $sl[$l]->complete = (boolean)$sline->complete;

                $sl[$l]->sealer = json_decode($sline->sealer);
                $sl[$l]->shipment = json_decode($sline->shipment);
        }
        return $sl;
    }

    /**
     * Select all drops.
     */
    public function selectAll()
    {
        $this->selected = [];
        foreach ($this->form->cp as $drop) {
            $this->selected[$drop['id']] = true;
        }
    }

	#[Computed]
	public function suppliers(){
		return Suppliers::allCached()->sortBy('name');
	}
    #[Computed(cache: true, key: 'factories-with-countries-wherehas-style_versions')]
	public function factories(){
		return Suppliers::whereHas('style_versions')->where('type', 'factory')->with('countries')->get();
	}
    #[Computed(cache: true, key: 'customers-wherehas-styles-sortedby-name')]
	public function customers(){
		return Customer::whereHas('styles')->get()->sortBy('name');
	}
    #[Computed]
	public function seasons(){
		return Seasons::allCached()->sortBy('description');
	}
	#[Computed]
	public function departments(){
		return Departments::allCached()->sortBy('name');
	}
	#[Computed]
	public function trucks(){
		return Shipment::allCached();
	}
}
