<?php

namespace App\Services\Matching;

use App\Models\StatementTransaction;
use App\Models\TransactionMatch;
use App\Services\AI\VertexClient;

/**
 * LLM-driven classification for statement transactions.
 * - Picks best department and account for the transaction
 * - Can propose splits when receipt lines indicate mixed departments
 * - Respects user preferences stored as rules (via ClassificationService)
 */
class TransactionClassificationService
{
    public function __construct(private VertexClient $vertex) {}

    /**
     * Classify a single statement transaction. If a receipt is matched,
     * use receipt meta and lines as additional context. If the model
     * proposes splits, store them under meta['splits'].
     */
    public function classify(StatementTransaction $transaction): void
    {
        $transaction->refresh();

        $receiptData = $this->collectReceiptContext($transaction);

        $model = config('vertex.models.classify', 'gemini-2.5-flash-lite');
        $system = 'You are a UK finance expense coding assistant. Given a statement transaction and optional matched receipt data, choose the most appropriate department and account code. If the receipt contains multiple line items that look like different departments, propose splits. Always output strict JSON. Keys: department (name), account (name), is_personal (boolean), splits (optional: array of {description, amount, department, account}). Use only the allowed values provided.';

        $payload = [
            'transaction' => [
                'merchant_name' => $transaction->merchant_name,
                'amount' => $transaction->amount,
                'currency' => $transaction->currency,
                'merchant_city' => $transaction->merchant_city,
                'merchant_state' => $transaction->merchant_state,
                'merchant_postcode' => $transaction->merchant_postcode,
                'mcc_description' => $transaction->mcc_description,
                'meta' => $transaction->meta ?? [],
            ],
            'matched_receipt' => $receiptData,
            'allowed_departments' => ClassificationService::allowedDepartments(),
            'allowed_accounts' => ClassificationService::allowedAccounts(),
        ];

        $resp = $this->vertex->generate($model, json_encode($payload), $system, ['responseMimeType' => 'application/json']);
        $json = $resp['json'];
        if (!is_array($json)) {
            return;
        }

        // Apply main classification
        $depId = ClassificationService::mapDepartmentNameToId($json['department'] ?? null);
        $accId = ClassificationService::mapAccountNameToId($json['account'] ?? null);
        if ($depId) $transaction->department_id = $depId;
        if ($accId) $transaction->account_id = $accId;
        if (array_key_exists('is_personal', $json)) {
            $transaction->is_personal = (bool) $json['is_personal'];
        }

        // Apply splits if provided
        if (!empty($json['splits']) && is_array($json['splits'])) {
            $splits = [];
            foreach ($json['splits'] as $split) {
                $sDepId = ClassificationService::mapDepartmentNameToId($split['department'] ?? null) ?? $depId;
                $sAccId = ClassificationService::mapAccountNameToId($split['account'] ?? null) ?? $accId;
                $amount = isset($split['amount']) ? (float) $split['amount'] : null;
                if ($amount === null || $amount <= 0) continue;
                $splits[] = [
                    'department_id' => $sDepId,
                    'account_id' => $sAccId,
                    'amount' => round($amount, 2),
                    'description' => $split['description'] ?? null,
                    'source' => 'llm_split',
                ];
            }

            if (!empty($splits)) {
                $meta = is_array($transaction->meta) ? $transaction->meta : (is_string($transaction->meta) ? json_decode($transaction->meta, true) : []);
                $meta['splits'] = $splits;
                $transaction->meta = $meta;
            }
        }

        $transaction->save();
    }

    private function collectReceiptContext(StatementTransaction $transaction): array
    {
        $matches = TransactionMatch::where('statement_transaction_id', $transaction->id)
            ->with('receipt.lines')
            ->get();

        if ($matches->isEmpty()) return [];

        $receipt = $matches->first()->receipt; // Prefer the best/first match
        if (!$receipt) return [];

        return [
            'merchant' => $receipt->merchant_name,
            'total_amount' => $receipt->total_amount,
            'currency' => $receipt->currency,
            'meta' => $receipt->meta ?? [],
            'lines' => $receipt->lines->map(function ($line) {
                $meta = is_array($line->meta) ? $line->meta : (is_string($line->meta) ? (json_decode($line->meta, true) ?: []) : []);
                $discounted = $meta['discount_info']['discounted_total'] ?? null;
                return [
                    'description' => $line->description,
                    'line_total' => $line->line_total,
                    'discounted_total' => $discounted,
                    'vat_rate' => $line->vat_rate,
                    'meta' => $meta,
                ];
            })->all(),
        ];
    }
}


