# 📋 Image Paste & Upload Guide

> **Implemented using the proven [StackOverflow solution](https://stackoverflow.com/questions/490908/paste-an-image-from-clipboard-using-javascript)** - the same method used by Gmail, Imgur, and other major web apps.

## ✨ The Simple 2-Step Process

### For Empty Colourways:

1. **Click the blue "Click & Paste" button** - this activates the paste area
2. **Press Ctrl+V** (or **⌘V** on Mac) - your image uploads instantly!

### For Colourways With Images:

1. **Click the blue clipboard icon (📋)** in the top-right corner - this activates paste mode
2. **Press Ctrl+V** (or **⌘V** on Mac) - replaces the existing image!

---

## 🎯 All Available Methods

### Method 1: Click & Paste (Recommended!)
```
1. Copy an image (screenshot, right-click → Copy Image, etc.)
2. Click "Click & Paste" button (or clipboard icon)
3. Card glows blue and shows "Ready! Press Ctrl+V"
4. Press Ctrl+V (⌘V on Mac)
5. ✓ Image uploads automatically!
```

**Why this works:** A hidden `contenteditable` div captures the paste event, extracts image data from `event.clipboardData.items`, and uploads it. This is the standard web approach - no browser permissions needed!

### Method 2: Choose File
```
1. Click "Choose File" button
2. Browse and select an image
3. ✓ Image uploads automatically!
```

### Method 3: Drag & Drop
```
1. Drag an image file from your computer
2. Drop it on any colourway card
3. ✓ Image uploads automatically!
```

**Bonus:** You can also drag images directly from web pages!

---

## 🔍 Technical Details (From StackOverflow)

The implementation uses the **paste event** with **clipboardData.items**:

```javascript
// Listen for paste event on contenteditable element
element.addEventListener('paste', function(event) {
    const items = event.clipboardData.items;
    
    for (let i = 0; i < items.length; i++) {
        const item = items[i];
        
        // Check if it's a file and an image
        if (item.kind === 'file' && item.type.startsWith('image/')) {
            const blob = item.getAsFile();
            // Upload the blob!
        }
    }
});
```

### Why This Approach?

- ✅ **No permissions required** - Uses standard paste event
- ✅ **Works everywhere** - Chrome, Firefox, Safari, Edge
- ✅ **Battle-tested** - Same method used by Gmail, Imgur, Pasteboard
- ✅ **Reliable** - Proven solution with 125k+ Stack Overflow views
- ✅ **Screenshots work** - Captures bitmap data directly from clipboard

---

## 🌐 Browser Compatibility

| Browser | Support | Notes |
|---------|---------|-------|
| Chrome/Edge | ✅ Full | Excellent support |
| Firefox | ✅ Full | Works perfectly |
| Safari | ✅ Full | 13.1+ required |
| Opera | ✅ Full | Full support |

**Reference:** [StackOverflow - Paste an image from clipboard using JavaScript](https://stackoverflow.com/questions/490908/paste-an-image-from-clipboard-using-javascript)

---

## 📸 What You Can Paste

- ✅ **Screenshots** - Windows Snipping Tool, macOS Cmd+Shift+4, Print Screen
- ✅ **Copied images from web** - Right-click any image → "Copy Image"
- ✅ **Image editors** - Photoshop, GIMP, Figma, etc.
- ✅ **Other apps** - Paint, Preview, Photos, etc.
- ✅ **Screen capture tools** - Greenshot, Lightshot, Snagit, etc.

---

## 💡 Visual Feedback

| State | What You See |
|-------|--------------|
| **Normal** | Grey background with icon |
| **After clicking button** | Blue border + blue glow |
| **Tooltip appears** | "Ready! Press Ctrl+V (⌘V on Mac) to paste" |
| **Dragging over** | Blue highlight + slightly larger |
| **Uploading** | Console shows progress |
| **Success** | Image appears immediately! |

---

## 🐛 Troubleshooting

### "No image found in clipboard"
**Solution:** Make sure you've actually copied an image:
- Take a screenshot first, OR
- Right-click an image and select "Copy Image" (not "Copy Image Address")

### "Nothing happens when I press Ctrl+V"
**Solution:** 
1. Make sure you clicked the "Click & Paste" button first
2. Check that the card has a blue glow (showing it's active)
3. Look at browser console (F12) for debugging info

### "I clicked the button but no blue glow appears"
**Solution:**
- You must be in **Edit Mode** (click "Edit Item" button at top)
- Try clicking directly on the colourway card area

---

## 🔑 Quick Reference

| Task | How To |
|------|--------|
| **Paste to empty colourway** | Click "Click & Paste" → Press Ctrl+V |
| **Replace existing image** | Click clipboard icon (📋) → Press Ctrl+V |
| **Choose from files** | Click "Choose File" button |
| **Drag from computer** | Drag file onto any colourway |
| **Drag from web** | Drag image from browser onto card |
| **See what's in clipboard** | Open browser console (F12) after clicking button |

---

## ⚙️ How It Works Behind the Scenes

1. When you click "Click & Paste", it focuses a hidden `contenteditable` div
2. This div has a `@paste` event listener
3. When you press Ctrl+V, the paste event fires
4. JavaScript reads `event.clipboardData.items`
5. It loops through items checking `item.kind === 'file'` and `item.type.startsWith('image/')`
6. When found, it calls `item.getAsFile()` to get the blob
7. The blob is converted to a File object
8. Livewire uploads it to the server
9. Done!

**No clipboard API permissions needed** - the paste event is triggered by YOUR action (Ctrl+V), so browsers allow it!

---

**Need Help?**

1. Are you in **Edit Mode**? (Click "Edit Item" at the top)
2. Did you copy an image to your clipboard?
3. Did you click the button first to activate it?
4. Open browser console (F12) to see detailed logs
5. Check the console for "Paste event triggered" message

---

*Implementation based on the [accepted StackOverflow answer](https://stackoverflow.com/questions/490908/paste-an-image-from-clipboard-using-javascript) with 125,000+ views and battle-tested in production.*
