<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Services\AI\ReceiptDeduplicationService;
use Illuminate\Console\Command;

class TestGenericSplitDetection extends Command
{
    protected $signature = 'test:generic-split-detection';
    protected $description = 'Test generic split receipt detection for any merchant';

    public function handle()
    {
        $this->info('Testing Generic Split Receipt Detection...');
        
        // Get some unmatched receipts to test with
        $unmatchedReceipts = Receipt::whereNull('receipt_group_id')
            ->whereDoesntHave('matches')
            ->with(['lines'])
            ->orderBy('receipt_date', 'desc')
            ->take(20)
            ->get();

        if ($unmatchedReceipts->count() < 2) {
            $this->warn('Not enough unmatched receipts to test. Need at least 2 receipts.');
            return;
        }

        $this->info("Found {$unmatchedReceipts->count()} unmatched receipts:");
        
        foreach ($unmatchedReceipts as $receipt) {
            $this->line("  ID: {$receipt->id} | {$receipt->merchant_name} | £{$receipt->total_amount} | {$receipt->receipt_date} | Lines: {$receipt->lines->count()}");
        }

        $this->info("\nTesting generic split detection...");
        
        $deduplicationService = app(ReceiptDeduplicationService::class);
        $results = [];
        
        foreach ($unmatchedReceipts as $receipt) {
            $this->info("Processing receipt {$receipt->id} ({$receipt->merchant_name})...");
            
            $group = $deduplicationService->processReceipt($receipt);
            
            if ($group) {
                $this->info("  ✅ Grouped! Group ID: {$group->id} - {$group->name}");
                $results[] = [
                    'receipt_id' => $receipt->id,
                    'group_id' => $group->id,
                    'group_name' => $group->name
                ];
            } else {
                $this->warn("  ❌ No grouping found");
            }
        }

        if (!empty($results)) {
            $this->info("\nGrouping Results:");
            foreach ($results as $result) {
                $this->line("  Receipt {$result['receipt_id']} → Group {$result['group_id']} ({$result['group_name']})");
            }
        } else {
            $this->warn("No groupings were created.");
        }

        // Show all existing groups
        $existingGroups = \App\Models\ReceiptGroup::with(['receipts'])->get();

        if ($existingGroups->count() > 0) {
            $this->info("\nAll existing groups:");
            foreach ($existingGroups as $group) {
                $this->line("  Group {$group->id}: {$group->name} ({$group->receipts->count()} receipts)");
                foreach ($group->receipts as $receipt) {
                    $this->line("    - Receipt {$receipt->id}: {$receipt->merchant_name} (£{$receipt->total_amount})");
                }
            }
        }

        $this->info("\nGeneric split detection test completed!");
        $this->info("The system now works for ANY merchant, not just hardcoded patterns!");
    }
}
