const crypto = require('crypto');
const db = require('./db-multi-tenant');

// Session configuration
const SESSION_COOKIE_NAME = 'vybe_session';
const SESSION_DURATION_DAYS = 30;

/**
 * Generate a secure session ID
 */
function generateSessionId() {
  return crypto.randomBytes(32).toString('hex');
}

/**
 * Create a new session for a user
 */
async function createUserSession(userId, req) {
  const sessionId = generateSessionId();
  const expiresAt = new Date(Date.now() + SESSION_DURATION_DAYS * 24 * 60 * 60 * 1000);
  const userAgent = req.headers['user-agent'] || null;
  const ipAddress = req.ip || req.connection.remoteAddress || null;
  
  await db.createSession(userId, sessionId, expiresAt, userAgent, ipAddress);
  
  return sessionId;
}

/**
 * Middleware to authenticate user from session cookie
 */
async function authenticateUser(req, res, next) {
  const sessionId = req.cookies[SESSION_COOKIE_NAME];
  
  if (!sessionId) {
    req.user = null;
    return next();
  }
  
  try {
    const session = await db.getSession(sessionId);
    
    if (!session) {
      // Invalid or expired session
      res.clearCookie(SESSION_COOKIE_NAME);
      req.user = null;
      return next();
    }
    
    // Attach user to request
    req.user = {
      id: session.user_id,
      username: session.username,
      email: session.email,
      fullName: session.full_name,
      subscriptionTier: session.subscription_tier,
      subscriptionStatus: session.subscription_status,
      trialEndsAt: session.trial_ends_at,
      subscriptionEndsAt: session.subscription_ends_at,
      currentPhotoCount: session.current_photo_count,
      maxPhotos: session.max_photos,
      currentStorageBytes: session.current_storage_bytes,
      maxStorageBytes: session.max_storage_bytes,
      siteName: session.site_name,
      primaryColor: session.primary_color,
      logoUrl: session.logo_url,
      hotChocolateDefaultUrl: session.hot_chocolate_default_url,
      hotChocolateText: session.hot_chocolate_text,
      contactEmail: session.contact_email,
      socialLinks: session.social_links || {},
      customDomain: session.custom_domain,
      customDomainVerified: session.custom_domain_verified,
      allowDownloads: session.allow_downloads,
      requireAgeVerification: session.require_age_verification
    };
    
    next();
  } catch (error) {
    console.error('Error authenticating user:', error);
    req.user = null;
    next();
  }
}

/**
 * Middleware to require authentication
 */
function requireAuth(req, res, next) {
  if (!req.user) {
    return res.redirect('/login');
  }
  next();
}

/**
 * Middleware to require that user is the tenant owner
 */
function requireTenantOwner(req, res, next) {
  if (!req.user) {
    return res.redirect('/login');
  }
  
  if (!req.tenant || req.user.id !== req.tenant.id) {
    return res.status(403).send('Access denied - you do not own this gallery');
  }
  
  next();
}

/**
 * Login a user
 */
async function loginUser(email, password, req, res) {
  const user = await db.verifyUserPassword(email, password);
  
  if (!user) {
    return null;
  }
  
  // Create session
  const sessionId = await createUserSession(user.id, req);
  
  // Set cookie
  res.cookie(SESSION_COOKIE_NAME, sessionId, {
    maxAge: SESSION_DURATION_DAYS * 24 * 60 * 60 * 1000,
    httpOnly: true,
    secure: process.env.NODE_ENV === 'production', // HTTPS only in production
    sameSite: 'lax'
  });
  
  return user;
}

/**
 * Logout a user
 */
async function logoutUser(req, res) {
  const sessionId = req.cookies[SESSION_COOKIE_NAME];
  
  if (sessionId) {
    await db.deleteSession(sessionId);
  }
  
  res.clearCookie(SESSION_COOKIE_NAME);
}

/**
 * Register a new user
 */
async function registerUser(username, email, password, fullName = null) {
  // Validate username (alphanumeric and hyphens only)
  const usernameRegex = /^[a-z0-9-]+$/;
  if (!usernameRegex.test(username)) {
    throw new Error('Username can only contain lowercase letters, numbers, and hyphens');
  }
  
  // Check minimum length
  if (username.length < 3 || username.length > 50) {
    throw new Error('Username must be between 3 and 50 characters');
  }
  
  // Check if username is taken
  const existingUsername = await db.getUserByUsername(username);
  if (existingUsername) {
    throw new Error('Username is already taken');
  }
  
  // Check if email is taken
  const existingEmail = await db.getUserByEmail(email);
  if (existingEmail) {
    throw new Error('Email is already registered');
  }
  
  // Validate email format
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  if (!emailRegex.test(email)) {
    throw new Error('Invalid email format');
  }
  
  // Validate password strength
  if (password.length < 8) {
    throw new Error('Password must be at least 8 characters long');
  }
  
  // Create user
  const user = await db.createUser(username, email, password, fullName);
  
  return user;
}

/**
 * Check if username is available
 */
async function isUsernameAvailable(username) {
  const user = await db.getUserByUsername(username);
  return !user;
}

module.exports = {
  authenticateUser,
  requireAuth,
  requireTenantOwner,
  loginUser,
  logoutUser,
  registerUser,
  isUsernameAvailable,
  SESSION_COOKIE_NAME
};

