# Stripe Price IDs Now Stored in Database

## Overview

Stripe Price IDs are now stored in the **database** instead of environment variables. This allows you to:

✅ Configure prices through the superuser interface  
✅ Update pricing without redeploying  
✅ Manage multiple tiers dynamically  
✅ Keep pricing configuration separate from code

## What Changed

### Before
```bash
# .env file
STRIPE_PRICE_STARTER=price_xxxxx
STRIPE_PRICE_PROFESSIONAL=price_xxxxx
STRIPE_PRICE_BUSINESS=price_xxxxx
```

Prices were hardcoded in environment variables and required a server restart to update.

### After
Prices are stored in the `subscription_tiers` database table and can be updated through:
1. **Superuser Interface** - Best for ongoing management
2. **SQL Commands** - Quick for initial setup
3. **API Endpoints** - For programmatic updates

## Database Schema

New table: `subscription_tiers`

```sql
CREATE TABLE subscription_tiers (
  id SERIAL PRIMARY KEY,
  tier_name TEXT NOT NULL UNIQUE,        -- 'starter', 'professional', 'business'
  display_name TEXT NOT NULL,            -- 'Starter', 'Professional', 'Business'
  description TEXT,
  
  -- Pricing
  price_monthly INTEGER NOT NULL,        -- Price in pence (1000 = £10.00)
  currency TEXT NOT NULL DEFAULT 'gbp',
  
  -- Stripe Configuration
  stripe_price_id TEXT,                  -- Stripe Price ID
  stripe_product_id TEXT,                -- Stripe Product ID
  
  -- Plan Limits
  max_photos INTEGER NOT NULL,
  max_storage_bytes BIGINT NOT NULL,
  max_albums INTEGER,                    -- NULL = unlimited
  custom_domain_enabled BOOLEAN,
  
  -- Trial Configuration
  trial_days INTEGER NOT NULL DEFAULT 0,
  
  -- Display and Status
  display_order INTEGER,
  is_active BOOLEAN,
  is_featured BOOLEAN,
  
  created_at TIMESTAMP,
  updated_at TIMESTAMP
);
```

## Files Modified

### 1. `migrations/004-add-subscription-tiers-table.sql` (NEW)
- Creates `subscription_tiers` table
- Inserts default tier configurations
- Sets up automatic timestamp updates

### 2. `src/db-multi-tenant.js`
Added functions:
- `getAllSubscriptionTiers()` - Get all active tiers
- `getSubscriptionTier(tierName)` - Get specific tier
- `updateSubscriptionTierConfig(tierName, updates)` - Update tier (superuser only)
- `getTierLimits(tierName)` - Get tier limits with database fallback

### 3. `src/stripe-service.js`
- Added `getSubscriptionTiers()` - Fetches tiers from database with 5-min cache
- Updated all functions to use database instead of `SUBSCRIPTION_TIERS` constant:
  - `createSubscription()`
  - `updateSubscription()`
  - `createCheckoutSession()`
  - `getTierConfig()`
  - `getAllTiers()`
  - `isConfigured()` - Now async, checks database for Price IDs
- Added `clearTierCache()` - Clear cache after updates

### 4. `src/server-multi-tenant.js`
- Updated all `stripeService.isConfigured()` calls to `await`
- Updated checkout API to fetch tier config from database
- Added validation for tier configuration

### 5. `STRIPE-SETUP-NOW.md`
- Updated setup instructions
- Removed environment variable config for Price IDs
- Added database migration steps
- Added SQL and superuser interface options

## Migration Steps

### 1. Run the Migration

```bash
# Apply the migration
psql photowebsite -f migrations/004-add-subscription-tiers-table.sql
```

This creates the table and inserts default configurations.

### 2. Add Your Stripe Price IDs

**Option A: Quick SQL Update**

```sql
UPDATE subscription_tiers 
SET stripe_price_id = 'price_1ABC123starter' 
WHERE tier_name = 'starter';

UPDATE subscription_tiers 
SET stripe_price_id = 'price_2DEF456professional' 
WHERE tier_name = 'professional';

UPDATE subscription_tiers 
SET stripe_price_id = 'price_3GHI789business' 
WHERE tier_name = 'business';
```

**Option B: Superuser Interface** (Recommended)

1. Login as superuser
2. Navigate to `/superuser/tiers`
3. Edit each tier and add Stripe Price ID
4. Save changes

### 3. Restart Server

```bash
npm start
```

### 4. Verify Configuration

Check that Stripe is configured:
```bash
curl http://localhost:3000/api/subscription/tiers
```

Should return tier configurations with Price IDs.

## Environment Variables

You still need these in `.env`:

```bash
# Required - Stripe API Keys
STRIPE_SECRET_KEY=sk_test_...
STRIPE_PUBLISHABLE_KEY=pk_test_...

# Optional - Fallback if database unavailable
# (Not recommended - use database instead)
STRIPE_PRICE_STARTER=price_...
STRIPE_PRICE_PROFESSIONAL=price_...
STRIPE_PRICE_BUSINESS=price_...
```

## Benefits

### 1. **Dynamic Configuration**
Update prices without touching code or environment variables.

### 2. **Centralized Management**
All pricing configuration in one place (database).

### 3. **Audit Trail**
Database tracks `updated_at` timestamp for price changes.

### 4. **Superuser Control**
Non-technical admins can update prices through UI.

### 5. **Environment Parity**
Same code works across dev/staging/prod with different database configs.

### 6. **Caching**
5-minute cache reduces database queries while staying fresh.

### 7. **Graceful Fallback**
If database fails, falls back to environment variables or defaults.

## API Usage

### Get All Tiers

```javascript
const tiers = await stripeService.getSubscriptionTiers();
console.log(tiers);
// {
//   starter: { name: 'Starter', price: 1000, priceId: 'price_...', ... },
//   professional: { name: 'Professional', price: 2500, priceId: 'price_...', ... },
//   business: { name: 'Business', price: 7500, priceId: 'price_...', ... }
// }
```

### Get Specific Tier

```javascript
const tier = await stripeService.getTierConfig('professional');
console.log(tier.priceId); // 'price_...'
```

### Check if Configured

```javascript
if (await stripeService.isConfigured()) {
  // Stripe is ready - all tiers have Price IDs
} else {
  // Missing Price IDs in database
}
```

### Clear Cache After Update

```javascript
// After updating tiers in database
stripeService.clearTierCache();
// Next call will fetch fresh data
```

## Troubleshooting

### Error: "Payment system not configured"

**Cause:** Missing Stripe Price IDs in database.

**Solution:**
1. Check database: `SELECT * FROM subscription_tiers;`
2. Verify `stripe_price_id` is set for all tiers
3. Run the UPDATE SQL commands
4. Restart server

### Error: "Invalid tier configuration"

**Cause:** Tier not found in database or missing Price ID.

**Solution:**
1. Verify tier exists: `SELECT * FROM subscription_tiers WHERE tier_name = 'professional';`
2. Check `is_active = true`
3. Ensure `stripe_price_id` is not NULL

### Prices Not Updating

**Cause:** 5-minute cache is still active.

**Solution:**
```javascript
// Clear cache programmatically
stripeService.clearTierCache();

// Or wait 5 minutes for cache to expire
```

### Database Connection Fails

**Cause:** Can't connect to database to fetch tiers.

**Solution:** System falls back to environment variables if available:
```bash
STRIPE_PRICE_STARTER=price_fallback_starter
STRIPE_PRICE_PROFESSIONAL=price_fallback_professional  
STRIPE_PRICE_BUSINESS=price_fallback_business
```

## Future Enhancements

Possible additions:

1. **Tier Management API** - REST API for tier CRUD operations
2. **Price History** - Track price changes over time
3. **A/B Testing** - Multiple price points per tier
4. **Regional Pricing** - Different prices by country/currency
5. **Promotional Pricing** - Temporary discounts and offers
6. **Annual Billing** - Add annual Price IDs alongside monthly

## Summary

✅ **Migration Created** - `004-add-subscription-tiers-table.sql`  
✅ **Database Functions Added** - Tier management in `db-multi-tenant.js`  
✅ **Stripe Service Updated** - Now reads from database with caching  
✅ **Server Updated** - Async `isConfigured()` checks  
✅ **Documentation Updated** - Setup guide reflects new approach  
✅ **Backwards Compatible** - Falls back to env vars if needed  

**Next Steps:** Run the migration and add your Stripe Price IDs!

