const db = require('./db-multi-tenant');

// Base domain configuration
const BASE_DOMAIN = process.env.BASE_DOMAIN || 'vybephoto.com';
const ADMIN_SUBDOMAIN = process.env.ADMIN_SUBDOMAIN || 'www';

/**
 * Extract subdomain from hostname
 */
function extractSubdomain(hostname) {
  // Remove port if present
  const hostWithoutPort = hostname.split(':')[0];
  
  // Check if this is a custom domain (doesn't end with BASE_DOMAIN)
  if (!hostWithoutPort.endsWith(BASE_DOMAIN)) {
    return { isCustomDomain: true, domain: hostWithoutPort, subdomain: null };
  }
  
  // Extract subdomain for BASE_DOMAIN
  const parts = hostWithoutPort.split('.');
  
  // If it's just the base domain (e.g., vybephoto.com) or www
  if (parts.length <= 2 || parts[0] === ADMIN_SUBDOMAIN) {
    return { isCustomDomain: false, domain: null, subdomain: null };
  }
  
  // Return the subdomain (first part)
  return { isCustomDomain: false, domain: null, subdomain: parts[0] };
}

/**
 * Middleware to detect and load tenant from subdomain or custom domain
 */
async function detectTenant(req, res, next) {
  const hostname = req.hostname || req.headers.host || '';
  const { isCustomDomain, domain, subdomain } = extractSubdomain(hostname);
  
  // Store raw detection results
  req.tenantInfo = {
    hostname,
    isCustomDomain,
    domain,
    subdomain,
    isMainSite: !isCustomDomain && !subdomain
  };
  
  // If this is the main site (www or base domain), no tenant to load
  if (req.tenantInfo.isMainSite) {
    req.tenant = null;
    return next();
  }
  
  try {
    let user = null;
    
    if (isCustomDomain) {
      // Look up user by custom domain
      user = await db.getUserByCustomDomain(domain);
      
      if (!user) {
        // Custom domain not found or not verified
        return res.status(404).send('Gallery not found. This custom domain may not be configured correctly.');
      }
    } else if (subdomain) {
      // Look up user by subdomain (username)
      user = await db.getUserByUsername(subdomain);
      
      if (!user) {
        return res.status(404).render('not-found', {
          pageTitle: 'Gallery Not Found',
          message: `No gallery found for "${subdomain}". Please check the URL.`,
          req: req
        });
      }
    }
    
    // Attach tenant (user) to request
    req.tenant = user;
    
    // Make tenant config available to views
    res.locals.tenant = user;
    res.locals.siteConfig = {
      siteName: user.site_name || user.username,
      primaryColor: user.primary_color || '#3b82f6',
      logoUrl: user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: user.contact_email || '',
      socialLinks: user.social_links || {},
      ageVerificationEnabled: user.require_age_verification || false
    };
    
    // Add RGB version of primary color for CSS variables
    res.locals.siteConfig.primaryColorRgb = hexToRgb(res.locals.siteConfig.primaryColor);
    
    next();
  } catch (error) {
    console.error('Error detecting tenant:', error);
    res.status(500).send('Error loading gallery');
  }
}

/**
 * Middleware to require a tenant (gallery must exist)
 */
function requireTenant(req, res, next) {
  if (!req.tenant) {
    return res.status(404).send('Gallery not found');
  }
  next();
}

/**
 * Helper to convert hex to RGB for CSS variables
 */
function hexToRgb(hex) {
  const shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;
  hex = hex.replace(shorthandRegex, function(m, r, g, b) {
    return r + r + g + g + b + b;
  });
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? `${parseInt(result[1], 16)}, ${parseInt(result[2], 16)}, ${parseInt(result[3], 16)}` : null;
}

/**
 * Helper to build gallery URL for a user
 */
function getGalleryUrl(user) {
  if (user.custom_domain && user.custom_domain_verified) {
    return `https://${user.custom_domain}`;
  }
  return `https://${user.username}.${BASE_DOMAIN}`;
}

module.exports = {
  detectTenant,
  requireTenant,
  extractSubdomain,
  getGalleryUrl,
  BASE_DOMAIN,
  ADMIN_SUBDOMAIN
};

