const db = require('./db-multi-tenant');

// Base domain configuration
const BASE_DOMAIN = process.env.BASE_DOMAIN || 'vybephoto.com';
const APP_SUBDOMAIN = process.env.APP_SUBDOMAIN || 'app'; // app.vybephoto.com = login/dashboard
const ADMIN_SUBDOMAIN = APP_SUBDOMAIN; // Legacy name for compatibility

/**
 * Extract subdomain from hostname
 */
function extractSubdomain(hostname) {
  // Remove port if present
  const hostWithoutPort = hostname.split(':')[0];
  
  // Special handling for localhost (development)
  if (hostWithoutPort === 'localhost' || hostWithoutPort === '127.0.0.1') {
    // Plain localhost = app subdomain (login/register area)
    return { isCustomDomain: false, domain: null, subdomain: APP_SUBDOMAIN, isAppSubdomain: true };
  }
  
  // Handle *.localhost (e.g., app.localhost, username.localhost)
  if (hostWithoutPort.endsWith('.localhost')) {
    const parts = hostWithoutPort.split('.');
    const subdomain = parts[0];
    
    // Check if it's the app subdomain
    if (subdomain === APP_SUBDOMAIN) {
      return { isCustomDomain: false, domain: null, subdomain: APP_SUBDOMAIN, isAppSubdomain: true };
    }
    
    // Otherwise it's a user gallery
    return { isCustomDomain: false, domain: null, subdomain: subdomain, isAppSubdomain: false };
  }
  
  // Check if this is a custom domain (doesn't end with BASE_DOMAIN)
  if (!hostWithoutPort.endsWith(BASE_DOMAIN)) {
    return { isCustomDomain: true, domain: hostWithoutPort, subdomain: null, isAppSubdomain: false };
  }
  
  // Extract subdomain for BASE_DOMAIN
  const parts = hostWithoutPort.split('.');
  
  // If it's just the base domain (e.g., vybephoto.com) or www
  if (parts.length <= 2 || parts[0] === 'www') {
    return { isCustomDomain: false, domain: null, subdomain: null, isAppSubdomain: false };
  }
  
  // Check if it's the app subdomain (app.vybephoto.com)
  if (parts[0] === APP_SUBDOMAIN) {
    return { isCustomDomain: false, domain: null, subdomain: APP_SUBDOMAIN, isAppSubdomain: true };
  }
  
  // Return the subdomain (first part) - this is a user gallery
  return { isCustomDomain: false, domain: null, subdomain: parts[0], isAppSubdomain: false };
}

/**
 * Middleware to detect and load tenant from subdomain or custom domain
 */
async function detectTenant(req, res, next) {
  const hostname = req.hostname || req.headers.host || '';
  const { isCustomDomain, domain, subdomain, isAppSubdomain } = extractSubdomain(hostname);
  
  // Store raw detection results
  req.tenantInfo = {
    hostname,
    isCustomDomain,
    domain,
    subdomain,
    isAppSubdomain,
    isMainSite: !isCustomDomain && !subdomain
  };
  
  // If this is the main site (www or base domain), no tenant to load
  // Main site should be handled by Joomla or external system
  if (req.tenantInfo.isMainSite) {
    req.tenant = null;
    return next();
  }
  
  // If this is the app subdomain (app.vybephoto.com), no tenant to load
  // This is the login/register/dashboard area
  if (isAppSubdomain) {
    req.tenant = null;
    return next();
  }
  
  try {
    let user = null;
    
    if (isCustomDomain) {
      // Look up user by custom domain
      user = await db.getUserByCustomDomain(domain);
      
      if (!user) {
        // Custom domain not found or not verified
        return res.status(404).send('Gallery not found. This custom domain may not be configured correctly.');
      }
    } else if (subdomain) {
      // Look up user by subdomain (username)
      user = await db.getUserByUsername(subdomain);
      
      if (!user) {
        return res.status(404).render('not-found', {
          pageTitle: 'Gallery Not Found',
          message: `No gallery found for "${subdomain}". Please check the URL.`,
          req: req
        });
      }
    }
    
    // Attach tenant (user) to request
    req.tenant = user;
    
    // Make tenant config available to views
    res.locals.tenant = user;
    res.locals.siteConfig = {
      siteName: user.site_name || user.username,
      primaryColor: user.primary_color || '#3b82f6',
      logoUrl: user.logo_url || '/nrw-web.png',
      hotChocolateDefaultUrl: user.hot_chocolate_default_url || 'https://square.link/u/NkgDiQCk',
      hotChocolateText: user.hot_chocolate_text || 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: user.contact_email || '',
      socialLinks: user.social_links || {},
      ageVerificationEnabled: user.require_age_verification || false
    };
    
    // Add RGB version of primary color for CSS variables
    res.locals.siteConfig.primaryColorRgb = hexToRgb(res.locals.siteConfig.primaryColor);
    
    next();
  } catch (error) {
    console.error('Error detecting tenant:', error);
    res.status(500).send('Error loading gallery');
  }
}

/**
 * Middleware to require a tenant (gallery must exist)
 */
function requireTenant(req, res, next) {
  if (!req.tenant) {
    return res.status(404).send('Gallery not found');
  }
  next();
}

/**
 * Helper to convert hex to RGB for CSS variables
 */
function hexToRgb(hex) {
  const shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;
  hex = hex.replace(shorthandRegex, function(m, r, g, b) {
    return r + r + g + g + b + b;
  });
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? `${parseInt(result[1], 16)}, ${parseInt(result[2], 16)}, ${parseInt(result[3], 16)}` : null;
}

/**
 * Helper to build gallery URL for a user
 */
function getGalleryUrl(user) {
  if (user.custom_domain && user.custom_domain_verified) {
    return `https://${user.custom_domain}`;
  }
  return `https://${user.username}.${BASE_DOMAIN}`;
}

module.exports = {
  detectTenant,
  requireTenant,
  extractSubdomain,
  getGalleryUrl,
  BASE_DOMAIN,
  APP_SUBDOMAIN,
  ADMIN_SUBDOMAIN // Legacy, same as APP_SUBDOMAIN
};

