<?php

namespace App\Models;

use DateTime;
use App\Models\Sizes;
use App\Casts\TitleCase;
use App\Models\BaseModel;
use App\Helper\Conversions;
use \Bkwld\Cloner\Cloneable;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Cache;
use App\Exceptions\DependencyException;
use OwenIt\Auditing\Contracts\Auditable;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class StyleVersions extends BaseModel implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use SoftDeletes;
    use HasFactory;
    use Cloneable;

    /**
     * The "booted" method of the model.
     * 
     * @return void
     */
    protected static function booted(): void
    {
        static::deleting(function ($model) {
            $model->colourways?->each?->delete();
            $model->prices?->each?->delete();
        });

        static::restoring(function ($model) {
            $model->colourways()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
            $model->prices()
                ->withTrashed()
                ->where('deleted_at', '>=', $model->deleted_at)
                ->each(function ($item, $key) { 
                    $item->restore(); 
                });
        });

        static::saved(function ($styleVersion) {
            if ($styleVersion->isDirty(['factory_id', 'gauge_id', 'yarn_ends'])) {
                $seasonId = $styleVersion->styles?->seasons_id;
                
                $cacheTags = ['prices', 'styleversion:' . $styleVersion->id];
                if ($seasonId) {
                    $cacheTags[] = 'season:' . $seasonId;
                }
                
                Cache::tags($cacheTags)->flush();
            }
            self::invalidatePriceResolutions($styleVersion);
        });

        static::deleted(function ($styleVersion) {
            self::invalidatePriceResolutions($styleVersion);
        });
    }

    protected $cloneable_relations = [
        'colourways',
    ];

    protected $casts = [
        'name'       => TitleCase::class,
        'yarn_ends'  => 'integer',
        'issued'     => 'datetime:Y-m-d',
        'created_at' => 'datetime:Y-m-d',
        'updated_at' => 'datetime:Y-m-d',
        'deleted_at' => 'datetime:Y-m-d',
        'sample_yarn_date' => 'datetime:Y-m-d',
        'sample_accessories_date' => 'datetime:Y-m-d',
    ];

	protected $fillable = [
        'styles_id',
        'factory_id',
        'name',
        'gauge_id',
        'created_at',
        'updated_at',
        'yarn_ends',
        'issued',
        'deleted_at',
        'sample_yarn',
        'sample_yarn_date',
        'sample_accessories',
        'sample_accessories_date',
        'comments'
    ];


    public function getImageAttribute(){
        // dd($this);
        if($this->colourways->whereNotNull('image')->first() != NULL)
            return url(asset('storage/' . $this->colourways->whereNotNull('image')->first()->image));
    }


































    public function cmt(){
        // Try to get CMT from price resolution cache first
        $priceResolution = \App\Models\PriceResolution::where('style_versions_id', $this->id)
            ->whereNull('colourways_id')
            ->whereNull('sizes_id')
            ->whereNull('phase_id')
            ->whereNotNull('cmt')
            ->first();
            
        if ($priceResolution) {
            return $priceResolution->cmt;
        }
        
        // Fallback to old method if no price resolution
        if($this->prices->where('cmt_status', '=', 'confirmed')->count() == 1){
            return $this->prices->sortBy('updated_at')->where('cmt_status', '=', 'confirmed')->first()->cmt;
        }
        else{
            $priceLast = 'NA';
            foreach($this->prices as $price){
                if($priceLast == 'NA')
                    $priceLast = $price->cmt;
                elseif($priceLast != $price->cmt)
                    return 'Many';
            }
            return $priceLast;
        }
    }
    public function cmt_if_confirmed(){
        if($this->prices->where('cmt_status', '=', 'confirmed')->count() == 1){
            return $this->prices->sortBy('updated_at')->where('cmt_status', '=', 'confirmed')->first()->cmt;
        }
    }







    public function styles()
    {
        return $this->belongsTo(Styles::class);
    }
    public function gauges()
    {
        return $this->belongsTo(Gauge::class, 'gauge_id');
    }
    public function factories()
    {
        return $this->belongsTo(Suppliers::class, 'factory_id');
    }

    public function prices()
    {
        return $this->hasMany(Price::class)
            ->orderBy(
                // A correlated subquery that fetches sizes.order
                Sizes::select('order')
                    ->whereColumn('sizes.id', 'prices.sizes_id')
            );
    }
	// public function samples()
    // {
    //     return $this->hasMany(Samples::class);
    // }
	public function colourways()
    {
        return $this->hasMany(Colourways::class);
    }













    protected static function invalidatePriceResolutions($styleVersion)
    {
        try {
            $service = app(\App\Services\PriceResolutionService::class);
            $service->invalidateAndWarmupByStyleVersion($styleVersion->id);
        } catch (\Exception $e) {
            \Log::error('Failed to invalidate price resolutions for style version', [
                'style_version_id' => $styleVersion->id,
                'error' => $e->getMessage()
            ]);
        }
    }

}
