#!/usr/bin/env node
/**
 * Payment Enforcement Script
 * 
 * This script checks all users for payment issues and automatically pauses accounts
 * that have been without valid payment for more than 7 days.
 * 
 * Rules:
 * - Superusers are exempt
 * - Users with skip_payment_enforcement = true are exempt
 * - Free/starter tier users are exempt
 * - Grace period is 7 days from when payment issue is first detected
 * 
 * Run this script daily via cron:
 * 0 2 * * * cd /path/to/VybePhoto && node enforce-payment.js >> logs/payment-enforcement.log 2>&1
 */

const db = require('./src/db-multi-tenant');
const stripeService = require('./src/stripe-service');

const GRACE_PERIOD_DAYS = 7;
const PAID_TIERS = ['professional', 'business'];

async function enforcePayment() {
  try {
    console.log('========================================');
    console.log('Payment Enforcement Check');
    console.log(new Date().toISOString());
    console.log('========================================\n');
    
    // Check if Stripe is configured
    const stripeConfigured = await stripeService.isConfigured();
    if (!stripeConfigured) {
      console.log('⚠️  Stripe not configured. Skipping payment enforcement.');
      return;
    }
    
    // Get all active users on paid tiers
    const result = await db.pool.query(
      `SELECT id, username, email, subscription_tier, subscription_status,
              stripe_customer_id, stripe_subscription_id,
              is_superuser, skip_payment_enforcement,
              last_payment_check_at, payment_grace_period_ends
       FROM users
       WHERE subscription_tier = ANY($1)
         AND subscription_status NOT IN ('paused', 'canceled', 'cancelled')
         AND is_superuser = FALSE
         AND skip_payment_enforcement = FALSE`,
      [PAID_TIERS]
    );
    
    const users = result.rows;
    console.log(`Found ${users.length} users on paid tiers to check.\n`);
    
    let checkedCount = 0;
    let pausedCount = 0;
    let graceCount = 0;
    let validCount = 0;
    
    for (const user of users) {
      checkedCount++;
      console.log(`[${checkedCount}/${users.length}] Checking user: ${user.username} (ID: ${user.id})`);
      
      try {
        // Check if user has Stripe customer ID
        if (!user.stripe_customer_id) {
          console.log(`  ⚠️  No Stripe customer ID`);
          await handlePaymentIssue(user, 'no_stripe_customer');
          if (shouldPauseUser(user)) {
            await pauseUser(user, 'No Stripe customer account');
            pausedCount++;
          } else {
            graceCount++;
          }
          continue;
        }
        
        // Check Stripe subscription status
        let hasValidPayment = false;
        
        if (user.stripe_subscription_id) {
          try {
            const subscription = await stripeService.stripe.subscriptions.retrieve(user.stripe_subscription_id);
            
            if (subscription.status === 'active' || subscription.status === 'trialing') {
              hasValidPayment = true;
            } else {
              console.log(`  ⚠️  Subscription status: ${subscription.status}`);
            }
          } catch (error) {
            console.log(`  ⚠️  Could not retrieve subscription: ${error.message}`);
          }
        } else {
          console.log(`  ⚠️  No Stripe subscription ID`);
        }
        
        if (hasValidPayment) {
          // Payment is valid - clear grace period if set
          if (user.payment_grace_period_ends) {
            await db.pool.query(
              `UPDATE users 
               SET payment_grace_period_ends = NULL,
                   last_payment_check_at = CURRENT_TIMESTAMP
               WHERE id = $1`,
              [user.id]
            );
            console.log(`  ✓ Valid payment - grace period cleared`);
          } else {
            await db.pool.query(
              'UPDATE users SET last_payment_check_at = CURRENT_TIMESTAMP WHERE id = $1',
              [user.id]
            );
            console.log(`  ✓ Valid payment`);
          }
          validCount++;
        } else {
          // Payment issue detected
          await handlePaymentIssue(user, 'invalid_payment');
          if (shouldPauseUser(user)) {
            await pauseUser(user, 'Payment issue detected for more than 7 days');
            pausedCount++;
          } else {
            graceCount++;
          }
        }
        
      } catch (error) {
        console.error(`  ❌ Error checking user ${user.username}:`, error.message);
      }
      
      console.log('');
    }
    
    console.log('========================================');
    console.log('Payment Enforcement Complete');
    console.log('========================================');
    console.log(`Total users checked: ${checkedCount}`);
    console.log(`Valid payments: ${validCount}`);
    console.log(`In grace period: ${graceCount}`);
    console.log(`Paused: ${pausedCount}`);
    console.log('========================================\n');
    
  } catch (error) {
    console.error('❌ Fatal error in payment enforcement:', error);
    process.exit(1);
  } finally {
    // Close database connection
    await db.pool.end();
  }
}

/**
 * Handle payment issue - set or update grace period
 */
async function handlePaymentIssue(user, reason) {
  if (!user.payment_grace_period_ends) {
    // First time detecting issue - set grace period
    const gracePeriodEnd = new Date();
    gracePeriodEnd.setDate(gracePeriodEnd.getDate() + GRACE_PERIOD_DAYS);
    
    await db.pool.query(
      `UPDATE users 
       SET payment_grace_period_ends = $1,
           last_payment_check_at = CURRENT_TIMESTAMP
       WHERE id = $2`,
      [gracePeriodEnd, user.id]
    );
    
    console.log(`  ⏰ Grace period set until: ${gracePeriodEnd.toLocaleString()}`);
    console.log(`     Reason: ${reason}`);
  } else {
    // Grace period already set - just update check time
    await db.pool.query(
      'UPDATE users SET last_payment_check_at = CURRENT_TIMESTAMP WHERE id = $1',
      [user.id]
    );
    
    const daysRemaining = Math.ceil((new Date(user.payment_grace_period_ends) - new Date()) / (1000 * 60 * 60 * 24));
    console.log(`  ⏰ Grace period expires in ${daysRemaining} days (${new Date(user.payment_grace_period_ends).toLocaleString()})`);
  }
}

/**
 * Check if user should be paused (grace period expired)
 */
function shouldPauseUser(user) {
  if (!user.payment_grace_period_ends) {
    return false; // No grace period set yet
  }
  
  const now = new Date();
  const graceEnd = new Date(user.payment_grace_period_ends);
  
  return now >= graceEnd;
}

/**
 * Pause user account
 */
async function pauseUser(user, reason) {
  await db.pool.query(
    `UPDATE users 
     SET subscription_status = 'paused',
         updated_at = CURRENT_TIMESTAMP
     WHERE id = $1`,
    [user.id]
  );
  
  console.log(`  ⏸️  PAUSED: ${user.username}`);
  console.log(`     Reason: ${reason}`);
  
  // TODO: Send email notification to user
  // await sendPausedNotificationEmail(user);
}

// Run the enforcement
enforcePayment()
  .then(() => {
    console.log('✅ Payment enforcement completed successfully');
    process.exit(0);
  })
  .catch((error) => {
    console.error('❌ Payment enforcement failed:', error);
    process.exit(1);
  });


