<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class Season extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'description',
        'start_date',
        'end_date',
        'image',
        'is_active',
        'sort_order',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($season) {
            if (empty($season->slug)) {
                $season->slug = Str::slug($season->name);
            }
        });
        
        static::updating(function ($season) {
            if ($season->isDirty('name') && empty($season->slug)) {
                $season->slug = Str::slug($season->name);
            }
        });
    }

    public function items(): HasMany
    {
        return $this->hasMany(Item::class);
    }

    public function getIsCurrentAttribute(): bool
    {
        $now = now()->toDateString();
        return $this->is_active && 
               $this->start_date <= $now && 
               ($this->end_date === null || $this->end_date >= $now);
    }

    public function getDurationAttribute(): string
    {
        if (!$this->start_date) {
            return 'No dates set';
        }
        
        if (!$this->end_date) {
            return 'Ongoing since ' . $this->start_date->format('M j, Y');
        }
        
        return $this->start_date->format('M j, Y') . ' - ' . $this->end_date->format('M j, Y');
    }
}
