<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Models\TransactionMatch;
use App\Services\AI\ReceiptDeduplicationService;
use App\Services\Matching\GroupedReceiptMatchingService;
use Illuminate\Console\Command;

class ReprocessReceiptsForGrouping extends Command
{
    protected $signature = 'receipts:reprocess-grouping {--force : Force reprocessing even if receipts are already matched}';
    protected $description = 'Reprocess existing receipts to find and group related receipts';

    public function handle(ReceiptDeduplicationService $deduplicationService, GroupedReceiptMatchingService $groupedMatching)
    {
        $this->info('Starting receipt grouping reprocessing...');
        
        $force = $this->option('force');
        
        // Get receipts to process
        $query = Receipt::query();
        
        if (!$force) {
            // Only process unmatched receipts by default
            $query->whereDoesntHave('matches');
        }
        
        $receipts = $query->whereNull('receipt_group_id')
            ->with(['lines'])
            ->orderBy('created_at', 'asc')
            ->get();

        if ($receipts->isEmpty()) {
            $this->warn('No receipts found to process.');
            return;
        }

        $this->info("Found {$receipts->count()} receipts to process");

        $progressBar = $this->output->createProgressBar($receipts->count());
        $progressBar->start();

        $groupedCount = 0;
        $matchedCount = 0;

        foreach ($receipts as $receipt) {
            try {
                // Check for relationships and group if found
                $group = $deduplicationService->processReceipt($receipt);
                
                if ($group) {
                    $groupedCount++;
                    $this->line("\nGrouped receipt {$receipt->id} into group {$group->id} ({$group->name})");
                    
                    // If this is a new group and we're forcing, try to match it
                    if ($force && $group->receipts->count() > 1) {
                        $groupedMatching->attemptMatchForGroup($group);
                        $matchedCount++;
                        $this->line("  → Attempted matching for group {$group->id}");
                    }
                }
                
                $progressBar->advance();
            } catch (\Exception $e) {
                $this->error("\nError processing receipt {$receipt->id}: " . $e->getMessage());
                $progressBar->advance();
            }
        }

        $progressBar->finish();
        $this->newLine(2);

        $this->info("Processing complete!");
        $this->info("Receipts grouped: {$groupedCount}");
        if ($force) {
            $this->info("Groups matched: {$matchedCount}");
        }

        // Show summary of created groups
        $groups = \App\Models\ReceiptGroup::withCount('receipts')->get();
        if ($groups->count() > 0) {
            $this->newLine();
            $this->info("Created groups:");
            foreach ($groups as $group) {
                $this->line("  - {$group->name} ({$group->receipts_count} receipts) - {$group->grouping_method}");
            }
        }
    }
}