<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Season;
use App\Models\Category;
use App\Models\Item;
use App\Models\Colourway;

class YarnProductsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create seasons if they don't exist
        $currentSeason = Season::firstOrCreate(
            ['slug' => 'current'],
            [
                'name' => 'Current Collection',
                'description' => 'Our current collection of premium yarns',
                'is_active' => true,
                'sort_order' => 1
            ]
        );

        // Create categories if they don't exist
        $cashmere = Category::firstOrCreate(
            ['slug' => 'cashmere'],
            [
                'name' => 'Cashmere',
                'description' => 'Luxurious cashmere yarns',
                'is_active' => true,
                'sort_order' => 1
            ]
        );

        $lambswool = Category::firstOrCreate(
            ['slug' => 'lambswool'],
            [
                'name' => 'Lambswool',
                'description' => 'Soft and warm lambswool yarns',
                'is_active' => true,
                'sort_order' => 2
            ]
        );

        $merino = Category::firstOrCreate(
            ['slug' => 'extrafine-merino'],
            [
                'name' => 'Extrafine Merino',
                'description' => 'Fine merino wool yarns',
                'is_active' => true,
                'sort_order' => 3
            ]
        );

        $cotton = Category::firstOrCreate(
            ['slug' => 'cotton'],
            [
                'name' => 'Cotton',
                'description' => 'Premium cotton yarns',
                'is_active' => true,
                'sort_order' => 4
            ]
        );

        $silk = Category::firstOrCreate(
            ['slug' => 'silk'],
            [
                'name' => 'Silk',
                'description' => 'Luxurious silk yarns',
                'is_active' => true,
                'sort_order' => 5
            ]
        );

        $blends = Category::firstOrCreate(
            ['slug' => 'blends'],
            [
                'name' => 'Blends',
                'description' => 'Premium yarn blends',
                'is_active' => true,
                'sort_order' => 6
            ]
        );

        $fancy = Category::firstOrCreate(
            ['slug' => 'fancy'],
            [
                'name' => 'Fancy',
                'description' => 'Specialty and fancy yarns',
                'is_active' => true,
                'sort_order' => 7
            ]
        );

        // Create products based on The Yarn Store
        $products = [
            [
                'name' => 'Pima 12',
                'description' => 'Premium Pima cotton yarn, perfect for delicate garments and accessories. Known for its exceptional softness and durability.',
                'short_description' => 'Premium Pima cotton yarn with exceptional softness',
                'category' => $cotton,
                'price' => 50.00,
                'sku' => 'PIMA12-001',
                'featured' => true,
                'colourways' => [
                    ['color_code' => '#FFFFFF', 'name' => 'White'],
                    ['color_code' => '#F5F5DC', 'name' => 'Cream'],
                    ['color_code' => '#F0E68C', 'name' => 'Khaki'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                    ['color_code' => '#000000', 'name' => 'Black'],
                ]
            ],
            [
                'name' => 'Cashwool 2/30 Mouline',
                'description' => 'Luxurious cashmere and wool blend with a subtle mouline twist. Perfect for elegant sweaters and accessories.',
                'short_description' => 'Cashmere wool blend with mouline twist',
                'category' => $blends,
                'price' => 53.10,
                'sku' => 'CW230M-001',
                'featured' => true,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#D2B48C', 'name' => 'Tan'],
                    ['color_code' => '#A0522D', 'name' => 'Sienna'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                    ['color_code' => '#2F4F4F', 'name' => 'Dark Slate'],
                ]
            ],
            [
                'name' => 'Cashwool 2/30',
                'description' => 'Classic cashmere and wool blend in a fine 2/30 count. Ideal for lightweight garments and delicate work.',
                'short_description' => 'Classic cashmere wool blend 2/30',
                'category' => $blends,
                'price' => 48.00,
                'sku' => 'CW230-001',
                'featured' => true,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Beige'],
                    ['color_code' => '#A0522D', 'name' => 'Sienna'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                ]
            ],
            [
                'name' => 'Pure Cashmere 2/28',
                'description' => '100% pure cashmere in 2/28 count. The ultimate luxury yarn for the most discerning knitters.',
                'short_description' => '100% pure cashmere luxury yarn',
                'category' => $cashmere,
                'price' => 85.00,
                'sku' => 'PC228-001',
                'featured' => false,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Camel'],
                    ['color_code' => '#A0522D', 'name' => 'Cognac'],
                    ['color_code' => '#8B4513', 'name' => 'Chocolate'],
                ]
            ],
            [
                'name' => 'Lambswool 2/28',
                'description' => 'Soft and warm lambswool in 2/28 count. Perfect for cozy sweaters and winter accessories.',
                'short_description' => 'Soft lambswool for winter garments',
                'category' => $lambswool,
                'price' => 35.00,
                'sku' => 'LW228-001',
                'featured' => false,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Beige'],
                    ['color_code' => '#A0522D', 'name' => 'Tan'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                ]
            ],
            [
                'name' => 'Extrafine Merino 2/30',
                'description' => 'Ultra-fine merino wool in 2/30 count. Known for its softness and excellent stitch definition.',
                'short_description' => 'Ultra-fine merino with excellent stitch definition',
                'category' => $merino,
                'price' => 42.00,
                'sku' => 'EM230-001',
                'featured' => false,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Beige'],
                    ['color_code' => '#A0522D', 'name' => 'Camel'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                ]
            ],
            [
                'name' => 'Silk Cone 2/20',
                'description' => 'Pure silk yarn on cone in 2/20 count. Perfect for luxurious shawls and delicate garments.',
                'short_description' => 'Pure silk on cone for luxury projects',
                'category' => $silk,
                'price' => 65.00,
                'sku' => 'SC220-001',
                'featured' => false,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Champagne'],
                    ['color_code' => '#A0522D', 'name' => 'Gold'],
                    ['color_code' => '#8B4513', 'name' => 'Bronze'],
                ]
            ],
            [
                'name' => 'Fancy Boucle 2/12',
                'description' => 'Textured boucle yarn in 2/12 count. Creates beautiful texture and visual interest in your projects.',
                'short_description' => 'Textured boucle for visual interest',
                'category' => $fancy,
                'price' => 38.00,
                'sku' => 'FB212-001',
                'featured' => false,
                'colourways' => [
                    ['color_code' => '#F5F5DC', 'name' => 'Natural'],
                    ['color_code' => '#F0E68C', 'name' => 'Cream'],
                    ['color_code' => '#D2B48C', 'name' => 'Beige'],
                    ['color_code' => '#A0522D', 'name' => 'Tan'],
                    ['color_code' => '#8B4513', 'name' => 'Brown'],
                ]
            ],
        ];

        foreach ($products as $productData) {
            $item = Item::firstOrCreate(
                ['slug' => \Illuminate\Support\Str::slug($productData['name'])],
                [
                    'name' => $productData['name'],
                    'description' => $productData['description'],
                    'short_description' => $productData['short_description'],
                    'seasons_id' => $currentSeason->id,
                    'categories_id' => $productData['category']->id,
                    'price' => $productData['price'],
                    'sku' => $productData['sku'],
                    'is_active' => true,
                    'featured' => $productData['featured'],
                    'no_columns' => $productData['no_columns'] ?? 3,
                    'specifications' => [
                        'weight' => '2/30',
                        'fiber_content' => $this->getFiberContent($productData['category']->name),
                        'care_instructions' => 'Hand wash cold, lay flat to dry',
                        'gauge' => '28-30 sts per 4 inches on US 3-4 needles',
                    ],
                ]
            );

            // Create colourways only if they don't exist
            foreach ($productData['colourways'] as $index => $colourwayData) {
                Colourway::firstOrCreate(
                    [
                        'items_id' => $item->id,
                        'reference' => $colourwayData['name'],
                    ],
                    [
                        'is_active' => true,
                        'sort_order' => $index + 1,
                    ]
                );
            }
        }
    }

    private function getFiberContent(string $categoryName): string
    {
        return match($categoryName) {
            'Cashmere' => '100% Cashmere',
            'Lambswool' => '100% Lambswool',
            'Extrafine Merino' => '100% Extrafine Merino',
            'Cotton' => '100% Pima Cotton',
            'Silk' => '100% Silk',
            'Blends' => '70% Cashmere, 30% Wool',
            'Fancy' => '100% Wool Boucle',
            default => '100% Natural Fiber',
        };
    }
}
