<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ClearAllData extends Command
{
    protected $signature = 'data:clear-all {--force : Skip confirmation}';
    protected $description = 'Clear all receipts, imports, queues, and related data for fresh testing';

    public function handle()
    {
        if (!$this->option('force')) {
            if (!$this->confirm('This will delete ALL receipts, statements, transactions, and related data. Are you sure?')) {
                $this->info('Operation cancelled.');
                return;
            }
        }

        $this->info('Clearing all data...');

        // Clear queues first
        $this->info('Clearing queues...');
        $this->call('queue:clear');
        $this->call('queue:clear', ['--queue' => 'default']);

        // For SQLite, we need to disable foreign key checks differently
        DB::statement('PRAGMA foreign_keys = OFF;');

        try {
            // Clear all data in correct order (respecting foreign keys)
            $this->info('Clearing transaction matches...');
            DB::table('transaction_matches')->delete();

            $this->info('Clearing receipt groups...');
            DB::table('receipt_groups')->delete();

            $this->info('Clearing receipt lines...');
            DB::table('receipt_lines')->delete();

            $this->info('Clearing receipts...');
            DB::table('receipts')->delete();

            $this->info('Clearing statement transactions...');
            DB::table('statement_transactions')->delete();

            $this->info('Clearing statements...');
            DB::table('statements')->delete();

            $this->info('Clearing VAT analyses...');
            DB::table('vat_analyses')->delete();

            $this->info('Clearing classification rules...');
            DB::table('classification_rules')->delete();

            $this->info('Clearing failed jobs...');
            DB::table('failed_jobs')->delete();

            $this->info('Clearing jobs...');
            DB::table('jobs')->delete();

        } finally {
            // Re-enable foreign key checks
            DB::statement('PRAGMA foreign_keys = ON;');
        }

        // Clear storage files
        $this->info('Clearing storage files...');
        $this->clearStorageFiles();

        // Reset auto-increment counters
        $this->info('Resetting auto-increment counters...');
        $this->resetAutoIncrement();

        $this->info('✅ All data cleared successfully!');
        $this->info('You can now test the receipt grouping pipeline from a clean state.');
    }

    private function clearStorageFiles()
    {
        try {
            // Clear receipt images
            $receiptPath = 'private/receipts';
            if (Storage::exists($receiptPath)) {
                $files = Storage::files($receiptPath);
                foreach ($files as $file) {
                    Storage::delete($file);
                }
                $this->line("  → Cleared receipt images");
            }

            // Clear any other temporary files
            $tempPath = 'private/temp';
            if (Storage::exists($tempPath)) {
                $files = Storage::allFiles($tempPath);
                foreach ($files as $file) {
                    Storage::delete($file);
                }
                $this->line("  → Cleared temporary files");
            }

        } catch (\Exception $e) {
            $this->warn("  → Warning: Could not clear all storage files: " . $e->getMessage());
        }
    }

    private function resetAutoIncrement()
    {
        $tables = [
            'receipts',
            'receipt_lines', 
            'receipt_groups',
            'statements',
            'statement_transactions',
            'transaction_matches',
            'vat_analyses',
            'classification_rules',
            'jobs',
            'failed_jobs'
        ];

        foreach ($tables as $table) {
            try {
                DB::statement("ALTER TABLE {$table} AUTO_INCREMENT = 1");
            } catch (\Exception $e) {
                // Ignore errors for tables that might not exist or don't have auto_increment
            }
        }
    }
}