# Testing Documentation

This document describes the comprehensive test suite for the Laravel Livewire Bootstrap Base application.

## Test Structure

The test suite is organized into the following test files:

### 1. Authentication Tests (`tests/Feature/AuthTest.php`)
Tests the complete authentication system including:
- Login page accessibility and form validation
- Registration functionality with proper validation
- Password hashing and security
- CSRF token protection
- Session management
- Logout functionality
- Protected route access control

### 2. User Role Tests (`tests/Feature/UserRoleTest.php`)
Tests the user role system including:
- Role attribute functionality
- Admin and user role checking methods
- Default role assignment for new users
- Role-based navigation display
- Role-based content display
- Role validation and updates

### 3. Livewire Tests (`tests/Feature/LivewireTest.php`)
Tests the Livewire component integration including:
- Welcome component rendering
- Authentication state handling
- User information display
- Component view integration
- Livewire styles and scripts inclusion
- Component data passing

### 4. Database Tests (`tests/Feature/DatabaseTest.php`)
Tests the database functionality including:
- Table structure validation
- Column existence verification
- User creation and role assignment
- Database connection testing
- Query functionality
- Data integrity and constraints

### 5. Bootstrap Tests (`tests/Feature/BootstrapTest.php`)
Tests the Bootstrap integration including:
- CSS and JS asset loading
- Bootstrap class usage
- Responsive design elements
- Form styling
- Navigation styling
- Component styling

### 6. Route Tests (`tests/Feature/RouteTest.php`)
Tests the routing system including:
- Route accessibility
- HTTP method validation
- Middleware functionality
- Route naming
- View rendering
- Error handling

### 7. Integration Tests (`tests/Feature/IntegrationTest.php`)
Tests the complete application workflow including:
- End-to-end user flows
- Cross-component integration
- Error handling scenarios
- Session management
- CSRF protection
- Middleware integration

## Running Tests

### Prerequisites
- PostgreSQL database running
- Test database `laravel_store_test` created
- All dependencies installed

### Quick Test Run
```bash
# Run all tests
php artisan test

# Run specific test file
php artisan test tests/Feature/AuthTest.php

# Run with coverage
php artisan test --coverage
```

### Using the Test Runner Script
```bash
# Make script executable (first time only)
chmod +x run-tests.sh

# Run all tests with detailed output
./run-tests.sh
```

### Individual Test Suites
```bash
# Authentication tests
php artisan test tests/Feature/AuthTest.php

# User role tests
php artisan test tests/Feature/UserRoleTest.php

# Livewire tests
php artisan test tests/Feature/LivewireTest.php

# Database tests
php artisan test tests/Feature/DatabaseTest.php

# Bootstrap tests
php artisan test tests/Feature/BootstrapTest.php

# Route tests
php artisan test tests/Feature/RouteTest.php

# Integration tests
php artisan test tests/Feature/IntegrationTest.php
```

## Test Configuration

### Database Configuration
Tests use a separate PostgreSQL database (`laravel_store_test`) to avoid conflicts with the main application database.

### Environment Variables
Test environment variables are configured in `phpunit.xml`:
- `APP_ENV=testing`
- `DB_CONNECTION=pgsql`
- `DB_DATABASE=laravel_store_test`
- `SESSION_DRIVER=array`
- `CACHE_STORE=array`

## Test Coverage

The test suite covers:

### ✅ Authentication System
- User registration and login
- Password hashing and validation
- Session management
- CSRF protection
- Route protection

### ✅ User Role System
- Role assignment and validation
- Role-based access control
- Admin and user privileges
- Role-based UI elements

### ✅ Livewire Integration
- Component rendering
- Authentication state handling
- Data binding
- View integration

### ✅ Database Functionality
- Table structure
- Data persistence
- Query functionality
- Constraints and validation

### ✅ Bootstrap Integration
- Asset loading
- CSS class usage
- Responsive design
- Form styling

### ✅ Routing System
- Route accessibility
- Middleware functionality
- HTTP method validation
- Error handling

### ✅ Integration Workflows
- Complete user journeys
- Cross-component functionality
- Error scenarios
- Security features

## Test Data

### Test Users
- **Admin User**: `admin@example.com` / `password` (role: admin)
- **Regular User**: `user@example.com` / `password` (role: user)

### Test Scenarios
- Valid and invalid login attempts
- User registration with various data
- Role-based access testing
- Session management
- CSRF protection
- Error handling

## Continuous Integration

The test suite is designed to run in CI/CD environments:
- Uses PostgreSQL for database testing
- Includes comprehensive error handling
- Covers all major functionality
- Provides detailed test output

## Adding New Tests

When adding new features, follow these guidelines:

1. **Create appropriate test files** for new functionality
2. **Use descriptive test names** that explain what is being tested
3. **Include both positive and negative test cases**
4. **Test edge cases and error scenarios**
5. **Update this documentation** when adding new test categories

## Test Maintenance

- Run tests regularly during development
- Update tests when changing functionality
- Ensure all tests pass before deployment
- Monitor test performance and optimize as needed

## Troubleshooting

### Common Issues

1. **Database Connection Errors**
   - Ensure PostgreSQL is running
   - Verify test database exists
   - Check database credentials

2. **Test Failures**
   - Check test database state
   - Verify test data setup
   - Review test assertions

3. **Performance Issues**
   - Use `RefreshDatabase` trait appropriately
   - Optimize test data creation
   - Consider test parallelization

### Getting Help

If you encounter issues with the test suite:
1. Check the test output for specific error messages
2. Verify all prerequisites are met
3. Review the test configuration
4. Check the Laravel testing documentation
