<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OrderLine extends Model
{
    protected $fillable = [
        'order_id',
        'item_id',
        'colourway_id',
        'quantity',
        'estimated_weight_g',
        'final_weight_g',
        'price_estimated_minor',
        'price_final_minor',
        'factory_po',
        'attributes',
        'sort_order',
    ];

    protected $casts = [
        'quantity' => 'integer',
        'estimated_weight_g' => 'integer',
        'final_weight_g' => 'integer',
        'price_estimated_minor' => 'integer',
        'price_final_minor' => 'integer',
        'attributes' => 'array',
        'sort_order' => 'integer',
    ];

    /**
     * Get the order that owns the order line.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the item that owns the order line.
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Get the colourway that owns the order line.
     */
    public function colourway(): BelongsTo
    {
        return $this->belongsTo(Colourway::class);
    }

    /**
     * Scope a query to only include order lines by order.
     */
    public function scopeByOrder($query, $orderId)
    {
        return $query->where('order_id', $orderId);
    }

    /**
     * Scope a query to only include order lines by item.
     */
    public function scopeByItem($query, $itemId)
    {
        return $query->where('item_id', $itemId);
    }

    /**
     * Scope a query to only include order lines by colourway.
     */
    public function scopeByColourway($query, $colourwayId)
    {
        return $query->where('colourway_id', $colourwayId);
    }

    /**
     * Get the estimated price in pounds.
     */
    public function getEstimatedPriceAttribute(): float
    {
        return $this->price_estimated_minor / 100;
    }

    /**
     * Get the final price in pounds.
     */
    public function getFinalPriceAttribute(): float
    {
        return $this->price_final_minor / 100;
    }

    /**
     * Get the estimated total price in pounds.
     */
    public function getEstimatedTotalAttribute(): float
    {
        return ($this->price_estimated_minor * $this->quantity) / 100;
    }

    /**
     * Get the final total price in pounds.
     */
    public function getFinalTotalAttribute(): float
    {
        return ($this->price_final_minor * $this->quantity) / 100;
    }

    /**
     * Get the estimated weight in kilograms.
     */
    public function getEstimatedWeightKgAttribute(): ?float
    {
        return $this->estimated_weight_g ? $this->estimated_weight_g / 1000 : null;
    }

    /**
     * Get the final weight in kilograms.
     */
    public function getFinalWeightKgAttribute(): ?float
    {
        return $this->final_weight_g ? $this->final_weight_g / 1000 : null;
    }
}
