<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Item;
use Illuminate\Support\Facades\Cookie;

class ProductDetail extends Component
{
    public $slug;
    public $product;
    public $selectedColourway = null;
    public $quantity = 1;

    public function mount($slug)
    {
        $this->slug = $slug;
        $this->product = Item::where('slug', $slug)
            ->with(['season', 'category', 'colourways' => function($query) {
                $query->orderBy('sort_order');
            }])
            ->firstOrFail();

        if ($this->product->colourways->count() > 0) {
            $this->selectedColourway = $this->product->colourways->first()->id;
        }
    }

    public function incrementQuantity()
    {
        $this->quantity++;
    }

    public function decrementQuantity()
    {
        if ($this->quantity > 1) {
            $this->quantity--;
        }
    }

    public function addToBasket()
    {
        $basket = json_decode(request()->cookie('basket', '[]'), true);
        
        $itemKey = $this->product->id . '-' . ($this->selectedColourway ?? 'default');
        
        $found = false;
        foreach ($basket as &$item) {
            if ($item['key'] === $itemKey) {
                $item['quantity'] += $this->quantity;
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            $basket[] = [
                'key' => $itemKey,
                'product_id' => $this->product->id,
                'colourway_id' => $this->selectedColourway,
                'quantity' => $this->quantity,
            ];
        }

        Cookie::queue('basket', json_encode($basket), 60 * 24 * 30);
        
        $this->dispatch('basket-updated');
        
        session()->flash('message', 'Added to basket!');
        $this->quantity = 1;
    }

    public function render()
    {
        $relatedProducts = Item::where('id', '!=', $this->product->id)
            ->where(function($query) {
                $query->where('categories_id', $this->product->categories_id)
                      ->orWhere('seasons_id', $this->product->seasons_id);
            })
            ->with(['colourways' => function($query) {
                $query->orderBy('sort_order');
            }])
            ->limit(4)
            ->get();

        return view('livewire.product-detail', [
            'relatedProducts' => $relatedProducts,
        ])->layout('layouts.shop');
    }
}

