<?php

namespace App\Console\Commands;

use App\Models\Receipt;
use App\Models\TransactionMatch;
use App\Services\AI\ReceiptDeduplicationService;
use App\Services\Matching\GroupedReceiptMatchingService;
use Illuminate\Console\Command;

class RegroupMatchedReceipts extends Command
{
    protected $signature = 'receipts:regroup-matched {--dry-run : Show what would be done without making changes}';
    protected $description = 'Find and regroup already matched receipts that should be grouped together';

    public function handle(ReceiptDeduplicationService $deduplicationService, GroupedReceiptMatchingService $groupedMatching)
    {
        $this->info('Starting regrouping of matched receipts...');
        
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->warn('DRY RUN MODE - No changes will be made');
        }

        // Get all matched receipts that aren't already grouped
        $matchedReceipts = Receipt::whereHas('matches')
            ->whereNull('receipt_group_id')
            ->with(['lines', 'matches.statementTransaction'])
            ->orderBy('created_at', 'asc')
            ->get();

        if ($matchedReceipts->isEmpty()) {
            $this->warn('No matched receipts found to process.');
            return;
        }

        $this->info("Found {$matchedReceipts->count()} matched receipts to analyze");

        $progressBar = $this->output->createProgressBar($matchedReceipts->count());
        $progressBar->start();

        $groupedCount = 0;
        $regroupedCount = 0;

        foreach ($matchedReceipts as $receipt) {
            try {
                // Check if this receipt should be grouped with others
                $group = $deduplicationService->processReceipt($receipt);
                
                if ($group) {
                    $groupedCount++;
                    $this->line("\nFound grouping for receipt {$receipt->id}");
                    $this->line("  → Would group into: {$group->name}");
                    $this->line("  → Group contains: {$group->receipts->count()} receipts");
                    
                    if (!$dryRun) {
                        // Check if all receipts in the group are matched to the same transaction
                        $transactionIds = $group->receipts->flatMap(fn($r) => $r->matches->pluck('statement_transaction_id'))->unique();
                        
                        if ($transactionIds->count() === 1) {
                            // All receipts are matched to the same transaction - this is good
                            $this->line("  → All receipts matched to same transaction - keeping existing matches");
                        } else {
                            // Mixed transactions - need to decide what to do
                            $this->line("  → Mixed transactions detected - keeping existing matches");
                        }
                        
                        $regroupedCount++;
                    }
                }
                
                $progressBar->advance();
            } catch (\Exception $e) {
                $this->error("\nError processing receipt {$receipt->id}: " . $e->getMessage());
                $progressBar->advance();
            }
        }

        $progressBar->finish();
        $this->newLine(2);

        $this->info("Analysis complete!");
        $this->info("Receipts that could be grouped: {$groupedCount}");
        
        if (!$dryRun) {
            $this->info("Receipts regrouped: {$regroupedCount}");
        } else {
            $this->warn('Run without --dry-run to apply changes');
        }

        // Show summary of potential groups
        if ($groupedCount > 0) {
            $this->newLine();
            $this->info("Potential groups found:");
            $groups = \App\Models\ReceiptGroup::withCount('receipts')->get();
            foreach ($groups as $group) {
                $this->line("  - {$group->name} ({$group->receipts_count} receipts) - {$group->grouping_method}");
            }
        }
    }
}