# Disabled Tier Management Guide

## Overview

This guide explains how disabled subscription tiers work in Vybe Photo. When a tier is disabled, existing users can continue using it while their subscription remains active, but new users cannot sign up for it.

## Key Features

### 1. **Existing Users Keep Access**
- Users on a disabled tier can continue using their current plan
- Their subscription limits and features remain intact
- Payment processing continues normally
- They can manage their payment methods and view their usage

### 2. **New Signups Prevented**
- Disabled tiers don't appear as options for new users
- Users cannot upgrade/downgrade to a disabled tier
- Stripe checkout sessions cannot be created for disabled tiers
- API calls to change to disabled tiers are rejected

### 3. **Visual Indicators**
- Users on disabled tiers see a "Legacy Plan" badge
- A warning message explains they can't switch back if they change plans
- The plan selection screen only shows active tiers

## Technical Implementation

### Database Schema

The `subscription_tiers` table includes an `is_active` boolean column:

```sql
is_active BOOLEAN NOT NULL DEFAULT TRUE
```

### Code Changes

#### 1. **Database Layer** (`src/db-multi-tenant.js`)

```javascript
async function getAllSubscriptionTiers(activeOnly = true) {
  const whereClause = activeOnly ? 'WHERE is_active = TRUE' : '';
  // Returns only active tiers by default
}
```

#### 2. **Stripe Service** (`src/stripe-service.js`)

- **`getSubscriptionTiers(activeOnly)`**: Gets tiers with optional filtering
  - `activeOnly = true` (default): Returns only active tiers for signup forms
  - `activeOnly = false`: Returns all tiers including disabled ones

- **`getTierConfig(tier)`**: Always returns tier config (including disabled tiers) for existing users

- **Protection against new signups to disabled tiers**:
  - `createSubscription()`: Validates tier is active
  - `updateSubscription()`: Validates tier is active
  - `createCheckoutSession()`: Validates tier is active

#### 3. **Subscription Routes** (`src/subscription-routes.js`)

- Validates tier changes against active tiers only
- Returns error if user tries to switch to disabled tier
- Passes both `availableTiers` (active) and `currentTierConfig` (can be disabled) to views

#### 4. **Subscription View** (`views/admin/subscription.ejs`)

- Dynamically renders only active tiers as options
- Shows "Legacy Plan" badge for disabled tiers
- Displays warning message about inability to switch back

## Usage Examples

### Disabling a Tier (Superuser Only)

1. Log in as superuser
2. Navigate to **Superuser Dashboard** → **Subscription Tiers**
3. Edit the tier you want to disable
4. Uncheck the "Active" checkbox
5. Save changes

### What Happens When a Tier is Disabled

**For Existing Users:**
- ✅ Continue using the tier normally
- ✅ Access all features and limits of the tier
- ✅ Renew subscription automatically
- ✅ Manage payment methods
- ⚠️ See "Legacy Plan" indicator
- ⚠️ Cannot switch back if they change plans

**For New Users:**
- ❌ Cannot select this tier during signup
- ❌ Cannot upgrade/downgrade to this tier
- ❌ Tier doesn't appear in pricing page

### Re-enabling a Tier

1. Log in as superuser
2. Navigate to **Superuser Dashboard** → **Subscription Tiers**
3. Edit the disabled tier
4. Check the "Active" checkbox
5. Save changes

The tier will immediately become available for new signups.

## Testing Checklist

### Testing Disabled Tiers

1. **Create a test user on a tier**
   ```bash
   # User signs up for "Professional" tier
   ```

2. **Disable the tier via superuser**
   ```bash
   # Superuser disables "Professional" tier
   ```

3. **Verify existing user can still use it**
   - [ ] User can log in
   - [ ] User sees correct usage limits
   - [ ] User can upload photos
   - [ ] User can create albums
   - [ ] User sees "Legacy Plan" badge
   - [ ] User sees warning about plan changes

4. **Verify new users cannot select it**
   - [ ] Tier doesn't appear in signup flow
   - [ ] API call to create subscription returns error
   - [ ] Checkout session creation fails

5. **Verify user cannot switch back to it**
   - [ ] User upgrades to "Business" tier
   - [ ] User tries to downgrade back to "Professional" (disabled)
   - [ ] Request is rejected with error

6. **Re-enable tier and verify**
   - [ ] Superuser enables "Professional" tier
   - [ ] Tier appears in pricing page
   - [ ] New users can sign up
   - [ ] Existing users can upgrade/downgrade to it

## Error Messages

| Error Code | Message | When It Occurs |
|------------|---------|----------------|
| `tier_unavailable` | Tier is not available for new subscriptions | User tries to switch to disabled tier |
| `invalid_tier` | Invalid tier requested | Tier doesn't exist in database |
| `same_tier` | Cannot change to the same tier | User tries to switch to current tier |

## Best Practices

### When to Disable a Tier

- **Pricing changes**: You want to offer a new pricing structure without affecting existing customers
- **Feature deprecation**: A tier includes features you no longer want to offer
- **Grandfathering**: Keep existing customers happy while moving to new plans
- **A/B testing**: Test new tiers without removing old ones

### When NOT to Disable a Tier

- **Temporary downtime**: Use maintenance mode instead
- **Bug fixes**: Fix the bug rather than disabling the tier
- **Force upgrades**: Better to communicate with users first

### Communication

When disabling a tier:
1. **Notify existing users** they're on a legacy plan
2. **Explain benefits** of staying vs switching
3. **Highlight new features** in active tiers
4. **Set a timeline** if you plan to sunset the tier eventually

## API Reference

### Get Available Tiers (Active Only)

```javascript
const activeTiers = await stripeService.getSubscriptionTiers(true);
// Returns: { starter: {...}, professional: {...}, business: {...} }
// Only includes tiers where is_active = true
```

### Get All Tiers (Including Disabled)

```javascript
const allTiers = await stripeService.getSubscriptionTiers(false);
// Returns all tiers regardless of is_active status
```

### Get Specific Tier Config

```javascript
const tierConfig = await stripeService.getTierConfig('professional');
// Returns tier config even if disabled (for existing users)
// Returns null if tier doesn't exist
```

### Validate Tier for New Subscription

```javascript
try {
  await stripeService.createSubscription(customerId, tier, trialDays);
  // Will throw error if tier is disabled
} catch (error) {
  // Error: "Tier professional is not available for new subscriptions"
}
```

## Database Queries

### Find Users on Disabled Tiers

```sql
SELECT u.id, u.username, u.email, u.subscription_tier, u.subscription_status
FROM users u
JOIN subscription_tiers st ON u.subscription_tier = st.tier_name
WHERE st.is_active = FALSE
  AND u.subscription_status IN ('active', 'trialing');
```

### Count Users Per Tier (Including Disabled)

```sql
SELECT 
  st.tier_name,
  st.is_active,
  COUNT(u.id) as user_count
FROM subscription_tiers st
LEFT JOIN users u ON u.subscription_tier = st.tier_name
  AND u.subscription_status IN ('active', 'trialing')
GROUP BY st.tier_name, st.is_active
ORDER BY st.display_order;
```

## Troubleshooting

### Issue: User on disabled tier cannot access features

**Solution**: Check that:
1. `getTierConfig()` is being used (not `getSubscriptionTiers(true)`)
2. User's `subscription_status` is 'active' or 'trialing'
3. User's `subscription_tier` field matches the tier name in database

### Issue: Users can still sign up for disabled tier

**Solution**: Clear the tier cache:
```javascript
stripeService.clearTierCache();
```

The cache refreshes every 5 minutes, or you can restart the server.

### Issue: Tier shows as disabled but should be active

**Solution**: Check the database:
```sql
UPDATE subscription_tiers 
SET is_active = TRUE 
WHERE tier_name = 'professional';
```

## See Also

- [Subscription Management Guide](./SUBSCRIPTION-UPGRADES-IMPLEMENTED.md)
- [Stripe Setup Guide](./STRIPE-SETUP-GUIDE.md)
- [Superuser Guide](./SUPERUSER-GUIDE.md)

